#include "system_info.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/stream/file.h>
#include <util/system/getpid.h>
#include <util/system/info.h>

namespace NPassport::NDaemon {
    TSystemInfo::TSystemInfo()
        : PageSize_(NSystemInfo::GetPageSize())
        , UnistatThreads_("proc.self.stat.num_threads", NUnistat::NSuffix::AVVV)
        , UnistatVMemory_("proc.self.stat.virtual_memory", NUnistat::NSuffix::AVVV)
        , UnistatRMemory_("proc.self.stat.resident_memory", NUnistat::NSuffix::AVVV)
    {
    }

    // http://man7.org/linux/man-pages/man5/proc.5.html
    static const char PROC_DELIMETER = ' ';
    static const TString PROC_PATH = "/proc/self/stat";
    void TSystemInfo::AddUnistat(NUnistat::TBuilder& builder) const {
        try {
            const TString stats = TUnbufferedFileInput(PROC_PATH).ReadAll();
            TStringBuf statsBuf(stats);

            size_t numThreads;
            size_t vsize;
            size_t rss;
            for (size_t idx = 0; idx < 19; ++idx) {
                statsBuf.NextTok(PROC_DELIMETER); // skip useless
            }

            Y_ENSURE(TryIntFromString<10>(statsBuf.NextTok(PROC_DELIMETER), numThreads),
                     "num_threads(20) is not number: " << stats);

            statsBuf.NextTok(PROC_DELIMETER); // skip useless
            statsBuf.NextTok(PROC_DELIMETER); // skip useless

            Y_ENSURE(TryIntFromString<10>(statsBuf.NextTok(PROC_DELIMETER), vsize),
                     "vsize(23) is not number: " << stats);
            Y_ENSURE(TryIntFromString<10>(statsBuf.NextTok(PROC_DELIMETER), rss),
                     "rss(24) is not number: " << stats);

            builder.AddRow(UnistatThreads_, numThreads);
            builder.AddRow(UnistatVMemory_, vsize);
            builder.AddRow(UnistatRMemory_, rss * PageSize_);
        } catch (const std::exception& e) {
            TLog::Warning() << "SystemInfo: failed to get data from /proc: " << e.what();
        }
    }
}
