#pragma once

#include "callback.h"
#include "system_info.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NDaemon::NPrivate {
    template <class T>
    class THasUnistatExt {
        using TYesType = char[1];
        using TNoType = char[2];

        template <typename C>
        static TYesType& Test(decltype(&C::AddUnistatExtended));
        template <typename C>
        static TNoType& Test(...);

    public:
        static constexpr bool value_ = sizeof(Test<T>(nullptr)) == sizeof(TYesType);
    };
}

namespace NPassport::NDaemon {
    class TBaseHandler {
    public:
        virtual ~TBaseHandler() = default;

        virtual void HandleRequest(NCommon::TRequest&) = 0;

        void Init(const TString&) {
        }
    };

    template <class T>
    class TUnistatHandler: public TBaseHandler {
    public:
        using TParentStats = std::function<void(NUnistat::TBuilder&)>;

        TUnistatHandler(T& s, TParentStats parentStats, const TString& path)
            : Service_(s)
            , ParentStats_(parentStats)
            , Path_(GetPath(path))
        {
        }

        void HandleRequest(NCommon::TRequest& req) override {
            const TString path = GetPath(req.GetPath());

            if (!path.StartsWith(Path_)) {
                req.SetStatus(HTTP_NOT_FOUND);
                req.Write(TString("Path is unknown: " + req.GetPath()));
                return;
            }

            TString stats;
            {
                NUnistat::TBuilder builder(stats);

                if (Path_ == path) {
                    AddUnistat(builder);
                } else if constexpr (NPrivate::THasUnistatExt<T>::value_) {
                    Service_.AddUnistatExtended(path.substr(Path_.size() - 1), builder);
                }
            }

            req.SetStatus(HTTP_OK);
            req.Write(stats);
        }

    private:
        void AddUnistat(NUnistat::TBuilder& builder) {
            builder.AddRow("log.debug_dmmm", TLog::GetCounters().Debug.GetValue());
            builder.AddRow("log.info_dmmm", TLog::GetCounters().Info.GetValue());
            builder.AddRow("log.warning_dmmm", TLog::GetCounters().Warning.GetValue());
            builder.AddRow("log.error_dmmm", TLog::GetCounters().Error.GetValue());

            ParentStats_(builder);
            Service_.AddUnistat(builder);
            SystemInfo_.AddUnistat(builder);
        }

        static TString GetPath(TString path) {
            if (path.back() != '/') {
                path.push_back('/');
            }
            return path;
        }

    private:
        T& Service_;
        TParentStats ParentStats_;
        const TString Path_;
        const TSystemInfo SystemInfo_;
    };
}
