#include "config.h"

#include "reader.h"

#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/rapidjson/include/rapidjson/pointer.h>

#include <util/generic/yexception.h>
#include <util/stream/file.h>

namespace NPassport::NJson {
    template <typename Target, typename From>
    static Target ToInt16(From from, const TString& jsonPoint) {
        static_assert(std::is_signed_v<Target> == std::is_signed_v<From>, "");

        Y_ENSURE_EX(from >= std::numeric_limits<Target>::min(),
                    TConfig::TBadValueException() << "jsonpoint is not i16: " << jsonPoint);
        Y_ENSURE_EX(from <= std::numeric_limits<Target>::max(),
                    TConfig::TBadValueException() << "jsonpoint is not i16: " << jsonPoint);

        return Target(from);
    }

    TConfig::TConfig(const TStringBuf body) {
        Y_ENSURE(NJson::TReader::DocumentAsObject(body, Doc_), "Failed to parse json object");
    }

    TConfig::TConfig(TConfig&&) noexcept = default;
    TConfig::~TConfig() = default;

    TConfig TConfig::ReadFromFile(const TString& file) {
        return ReadFromMemory(TFileInput(file).ReadAll());
    }

    TConfig TConfig::ReadFromMemory(const TStringBuf body) {
        return TConfig(body);
    }

    bool TConfig::Contains(const TString& jsonPoint) const {
        const rapidjson::Value* dummy;
        return GetOptionalValue(jsonPoint, dummy);
    }

    template <>
    bool TConfig::AsImpl<bool>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsBool(), TBadValueException() << "jsonpoint is not bool: " << jsonPoint);
        return value.GetBool();
    }

    template <>
    i16 TConfig::AsImpl<i16>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsInt(), TBadValueException() << "jsonpoint is not i16: " << jsonPoint);
        return ToInt16<i16>(value.GetInt(), jsonPoint);
    }

    template <>
    ui16 TConfig::AsImpl<ui16>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsUint(), TBadValueException() << "jsonpoint is not ui16: " << jsonPoint);
        return ToInt16<ui16>(value.GetUint(), jsonPoint);
    }

    template <>
    i32 TConfig::AsImpl<i32>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsInt(), TBadValueException() << "jsonpoint is not i32: " << jsonPoint);
        return value.GetInt();
    }

    template <>
    ui32 TConfig::AsImpl<ui32>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsUint(), TBadValueException() << "jsonpoint is not ui32: " << jsonPoint);
        return value.GetUint();
    }

    template <>
    i64 TConfig::AsImpl<i64>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsInt64(), TBadValueException() << "jsonpoint is not i64: " << jsonPoint);
        return value.GetInt64();
    }

    template <>
    ui64 TConfig::AsImpl<ui64>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsUint64(), TBadValueException() << "jsonpoint is not ui64: " << jsonPoint);
        return value.GetUint64();
    }

    template <>
    double TConfig::AsImpl<double>(const rapidjson::Value& value, const TString& jsonPoint) const {
        if (value.IsDouble()) {
            return value.GetDouble();
        }
        if (value.IsUint64()) {
            return value.GetUint64();
        }
        if (value.IsInt64()) {
            return value.GetInt64();
        }

        ythrow TBadValueException() << "jsonpoint is not double: " << jsonPoint;
    }

    template <>
    TString TConfig::AsImpl<TString>(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsString(), TBadValueException() << "jsonpoint is not string: " << jsonPoint);
        Y_ENSURE_EX(value.GetStringLength() > 0, TEmptyException() << "value is empty: " << jsonPoint);
        return TString(value.GetString(), value.GetStringLength());
    }

    template <>
    std::vector<i16> TConfig::AsImpl<std::vector<i16>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<i16>(value, jsonPoint);
    }

    template <>
    std::vector<ui16> TConfig::AsImpl<std::vector<ui16>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<ui16>(value, jsonPoint);
    }

    template <>
    std::vector<i32> TConfig::AsImpl<std::vector<i32>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<i32>(value, jsonPoint);
    }

    template <>
    std::vector<ui32> TConfig::AsImpl<std::vector<ui32>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<ui32>(value, jsonPoint);
    }

    template <>
    std::vector<i64> TConfig::AsImpl<std::vector<i64>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<i64>(value, jsonPoint);
    }

    template <>
    std::vector<ui64> TConfig::AsImpl<std::vector<ui64>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<ui64>(value, jsonPoint);
    }

    template <>
    std::vector<double> TConfig::AsImpl<std::vector<double>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<double>(value, jsonPoint);
    }

    template <>
    std::vector<TString> TConfig::AsImpl<std::vector<TString>>(const rapidjson::Value& value, const TString& jsonPoint) const {
        return AsArray<TString>(value, jsonPoint);
    }

    template <typename T>
    std::vector<T> TConfig::AsArray(const rapidjson::Value& value, const TString& jsonPoint) const {
        Y_ENSURE_EX(value.IsArray(), TBadValueException() << "jsonpoint is not an array: " << jsonPoint);
        std::vector<T> values;
        values.reserve(value.Size());
        for (size_t i = 0; i < value.Size(); ++i) {
            values.push_back(AsImpl<T>(value[i], NUtils::CreateStr(jsonPoint, '/', i)));
        }
        return values;
    }

    std::vector<TString> TConfig::SubKeys(const TString& jsonPoint) const {
        const rapidjson::Value* value;
        if (!GetOptionalValue(jsonPoint, value)) {
            return {};
        }

        Y_ENSURE(value->IsArray() || value->IsObject(),
                 "jsonpoint is not array or object: " << jsonPoint);

        std::vector<TString> res;

        if (value->IsArray()) {
            for (size_t idx = 0; idx < value->GetArray().Size(); ++idx) {
                res.push_back(NUtils::CreateStr(jsonPoint, "/", idx));
            }
        } else {
            for (auto it = value->MemberBegin(); it != value->MemberEnd(); ++it) {
                res.push_back(NUtils::CreateStr(
                    jsonPoint,
                    "/",
                    TStringBuf(it->name.GetString(), it->name.GetStringLength())));
            }
        }

        return res;
    }

    std::unique_ptr<NUtils::TFileLogger> TConfig::CreateLogger(const TString& jsonPoint) const {
        TString filename = As<TString>(jsonPoint + "/file");
        TString level = As<TString>(jsonPoint + "/level", "DEBUG");
        bool printLevel = As<bool>(jsonPoint + "/print_level", false);
        TString timeFormat = As<TString>(jsonPoint + "/time_format", "");
        int linesPerShot = As<ui64>(jsonPoint + "/lines_per_shot", 1024);

        return std::make_unique<NUtils::TFileLogger>(filename, level, printLevel, timeFormat, linesPerShot);
    }

    void TConfig::InitCommonLog(const TString& jsonPoint) const {
        const TString fileName = As<TString>(jsonPoint + "/file");
        if (fileName != "_NOLOG_") {
            TLog::Init(CreateLogger(jsonPoint));
        }
    }

    TString TConfig::GetKeyFromPath(TStringBuf path) {
        return TString(path.RNextTok('/'));
    }

    const rapidjson::Value& TConfig::GetValueStrict(const TString& jsonPoint) const {
        const rapidjson::Value* value = rapidjson::Pointer(jsonPoint.c_str()).Get(Doc_);
        Y_ENSURE_EX(value, TMissingException() << "jsonpoint not found: " << jsonPoint);
        Y_ENSURE_EX(!value->IsNull(), TMissingException() << "jsonpoint is null: " << jsonPoint);
        return *value;
    }

    bool TConfig::GetOptionalValue(const TString& jsonPoint, const rapidjson::Value*& value) const {
        value = rapidjson::Pointer(jsonPoint.c_str()).Get(Doc_);
        return value && !value->IsNull();
    }
}
