#pragma once

#include <contrib/libs/rapidjson/include/rapidjson/document.h>

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>

#include <memory>
#include <vector>

namespace NPassport::NUtils {
    class TFileLogger;
}

namespace NPassport::NJson {
    class TConfig: TMoveOnly {
    public:
        class TMissingException: public yexception {
        };
        class TEmptyException: public yexception {
        };
        class TBadValueException: public yexception {
        };

    public:
        TConfig(TConfig&&) noexcept;
        ~TConfig();

        static TConfig ReadFromFile(const TString& file);
        static TConfig ReadFromMemory(const TStringBuf body);

        bool Contains(const TString& jsonPoint) const;

        template <typename T>
        T As(const TString& jsonPoint) const {
            return AsImpl<T>(GetValueStrict(jsonPoint), jsonPoint);
        }

        template <typename T>
        T As(const TString& jsonPoint, T defval) const {
            const rapidjson::Value* value;
            try {
                return GetOptionalValue(jsonPoint, value) ? AsImpl<T>(*value, jsonPoint) : defval;
            } catch (const TEmptyException&) {
                return defval;
            }
        }

        std::vector<TString> SubKeys(const TString& jsonPoint) const;

        std::unique_ptr<NUtils::TFileLogger> CreateLogger(const TString& jsonPoint) const;
        void InitCommonLog(const TString& jsonPoint) const;

        static TString GetKeyFromPath(TStringBuf path);

    private:
        TConfig(const TStringBuf body);

        const rapidjson::Value& GetValueStrict(const TString& jsonPoint) const;
        bool GetOptionalValue(const TString& jsonPoint, const rapidjson::Value*& value) const;

        template <typename T>
        T AsImpl(const rapidjson::Value&, const TString&) const {
            struct TDummy;
            static_assert(std::is_same_v<TDummy, T>, "Type T is not supported");
        }

        template <>
        bool AsImpl<bool>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        i16 AsImpl<i16>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        ui16 AsImpl<ui16>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        i32 AsImpl<i32>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        ui32 AsImpl<ui32>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        i64 AsImpl<i64>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        ui64 AsImpl<ui64>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        double AsImpl<double>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        TString AsImpl<TString>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<i16> AsImpl<std::vector<i16>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<ui16> AsImpl<std::vector<ui16>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<i32> AsImpl<std::vector<i32>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<ui32> AsImpl<std::vector<ui32>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<i64> AsImpl<std::vector<i64>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<ui64> AsImpl<std::vector<ui64>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<double> AsImpl<std::vector<double>>(const rapidjson::Value& value, const TString& jsonPoint) const;
        template <>
        std::vector<TString> AsImpl<std::vector<TString>>(const rapidjson::Value& value, const TString& jsonPoint) const;

        template <typename T>
        std::vector<T> AsArray(const rapidjson::Value& value, const TString& jsonPoint) const;

    private:
        rapidjson::Document Doc_;
    };
}
