#pragma once

#include <contrib/libs/rapidjson/include/rapidjson/document.h>

#include <util/generic/strbuf.h>
#include <util/generic/string.h>

namespace NPassport::NJson {
    class TReader {
    public:
        static bool DocumentAsArray(const TStringBuf str, rapidjson::Document& doc) {
            return !doc.Parse(str.data(), str.size()).HasParseError() && doc.IsArray();
        }

        static bool DocumentAsObject(const TStringBuf str, rapidjson::Document& doc) {
            return !doc.Parse(str.data(), str.size()).HasParseError() && doc.IsObject();
        }

        static bool Member(const rapidjson::Value& obj, const char* key, const rapidjson::Value*& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }

            value = &it->value;
            return true;
        }

        static bool MemberAsString(const rapidjson::Value& obj, const char* key, const char*& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsString()) {
                return false;
            }

            value = rapidValue.GetString();
            return true;
        }

        template <class Str>
        static bool MemberAsString(const rapidjson::Value& obj, const char* key, Str& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsString()) {
                return false;
            }

            value = Str(rapidValue.GetString(), rapidValue.GetStringLength());
            return true;
        }

        static bool MemberAsArray(const rapidjson::Value& obj, const char* key, const rapidjson::Value*& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsArray()) {
                return false;
            }

            value = &it->value;
            return true;
        }

        static bool MemberAsObject(const rapidjson::Value& obj, const char* key, const rapidjson::Value*& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsObject()) {
                return false;
            }

            value = &it->value;
            return true;
        }

        static bool MemberAsInt(const rapidjson::Value& obj, const char* key, int& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsInt()) {
                return false;
            }

            value = it->value.GetInt();
            return true;
        }

        static bool MemberAsUInt(const rapidjson::Value& obj, const char* key, ui32& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsUint()) {
                return false;
            }

            value = it->value.GetUint();
            return true;
        }

        static bool MemberAsInt64(const rapidjson::Value& obj, const char* key, i64& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsInt64()) {
                return false;
            }

            value = it->value.GetInt64();
            return true;
        }

        static bool MemberAsUInt64(const rapidjson::Value& obj, const char* key, ui64& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsUint64()) {
                return false;
            }

            value = it->value.GetUint64();
            return true;
        }

        static bool MemberAsBool(const rapidjson::Value& obj, const char* key, bool& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsBool()) {
                return false;
            }

            value = it->value.GetBool();
            return true;
        }

        static bool MemberAsDouble(const rapidjson::Value& obj, const char* key, double& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;

            if (rapidValue.IsDouble()) {
                value = it->value.GetDouble();
                return true;
            }

            if (rapidValue.IsUint64()) {
                value = it->value.GetUint64();
                return true;
            }

            if (rapidValue.IsInt64()) {
                value = it->value.GetInt64();
                return true;
            }

            return false;
        }

        static bool MemberAsNull(const rapidjson::Value& obj, const char* key, bool& value) {
            if (!obj.IsObject()) {
                return false;
            }

            auto it = obj.FindMember(key);
            if (it == obj.MemberEnd()) {
                return false;
            }
            const rapidjson::Value& rapidValue = it->value;
            if (!rapidValue.IsNull()) {
                return false;
            }

            value = true;
            return true;
        }
    };

}
