#include <passport/infra/libs/cpp/json/config.h>

#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>

using namespace NPassport::NJson;
using namespace NPassport::NUtils;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspJsonConfig) {
    Y_UNIT_TEST(badconfigs) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TConfig::ReadFromFile("/"), TFileError, R"(can not read data from)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TConfig::ReadFromMemory("{"), yexception, "Failed to parse json object");
    }

    Y_UNIT_TEST(jsonPointAccess) {
        const TConfig conf = TConfig::ReadFromMemory(
            R"({
                "null_value": null,
                "string_value": "some_str",
                "empty_string_value": "",
                "int_value": 42,
                "double_value": 0.25,
                "bool_value": true,
                "array_value": [
                    {
                        "val":100500
                    },
                    {
                        "val":"100501"
                    }
                ],
                "subkeys_value": {
                    "key1": 42,
                    "key2": false
                }
            })");

        UNIT_ASSERT(!conf.Contains("/! root"));
        UNIT_ASSERT(!conf.Contains("/missing_value"));
        UNIT_ASSERT(!conf.Contains("/null_value"));
        UNIT_ASSERT(conf.Contains("/int_value"));
        UNIT_ASSERT(conf.Contains("/string_value"));
        UNIT_ASSERT(conf.Contains("/double_value"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/missing_value"),
                                       TConfig::TMissingException,
                                       "jsonpoint not found: /missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/null_value"),
                                       TConfig::TMissingException,
                                       "jsonpoint is null: /null_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/int_value"),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not string: /int_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/empty_string_value"),
                                       TConfig::TEmptyException,
                                       "value is empty: /empty_string_value");
        UNIT_ASSERT_VALUES_EQUAL("some_str", conf.As<TString>("/string_value"));

        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.As<TString>("/missing_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.As<TString>("/null_value", "def_val"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/int_value"),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not string: /int_value");
        UNIT_ASSERT_VALUES_EQUAL("some_str", conf.As<TString>("/string_value", "def_val"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/missing_value"),
                                       TConfig::TMissingException,
                                       "jsonpoint not found: /missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/null_value"),
                                       TConfig::TMissingException,
                                       "jsonpoint is null: /null_value");
        UNIT_ASSERT_VALUES_EQUAL(42, conf.As<int>("/int_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/string_value"),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not i32: /string_value");

        UNIT_ASSERT_VALUES_EQUAL(666, conf.As<int>("/missing_value", 666));
        UNIT_ASSERT_VALUES_EQUAL(666, conf.As<int>("/null_value", 666));
        UNIT_ASSERT_VALUES_EQUAL(42, conf.As<int>("/int_value", 666));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/string_value", 666),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not i32: /string_value");

        UNIT_ASSERT(conf.As<bool>("/bool_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<bool>("/missing_value"),
                                       TConfig::TMissingException,
                                       "jsonpoint not found: /missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<bool>("/int_value"),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not bool: /int_value");

        UNIT_ASSERT(conf.As<bool>("/bool_value", false));
        UNIT_ASSERT(!conf.As<bool>("/missing_value", false));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<bool>("/int_value"),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not bool: /int_value");

        UNIT_ASSERT_VALUES_EQUAL(std::vector<TString>{}, conf.SubKeys("/! root"));
        UNIT_ASSERT_VALUES_EQUAL(std::vector<TString>{}, conf.SubKeys("/missing_value"));
        UNIT_ASSERT_VALUES_EQUAL(std::vector<TString>{}, conf.SubKeys("/null_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.SubKeys("/int_value"),
                                       yexception,
                                       "jsonpoint is not array or object: /int_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.SubKeys("/string_value"),
                                       yexception,
                                       "jsonpoint is not array or object: /string_value");
        std::vector<TString> vec = conf.SubKeys("/array_value");
        UNIT_ASSERT_VALUES_EQUAL(std::vector<TString>({
                                     "/array_value/0",
                                     "/array_value/1",
                                 }),
                                 vec);
        UNIT_ASSERT_VALUES_EQUAL(100500, conf.As<ui32>(vec[0] + "/val"));
        UNIT_ASSERT_VALUES_EQUAL("100501", conf.As<TString>(vec[1] + "/val"));

        vec = conf.SubKeys("/subkeys_value");
        UNIT_ASSERT_VALUES_EQUAL(std::vector<TString>({
                                     "/subkeys_value/key1",
                                     "/subkeys_value/key2",
                                 }),
                                 vec);
        UNIT_ASSERT_VALUES_EQUAL(42, conf.As<ui32>(vec[0]));
        UNIT_ASSERT_VALUES_EQUAL(false, conf.As<bool>(vec[1]));

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<i16>("/array_value/0"),
                                       TConfig::TBadValueException,
                                       "jsonpoint is not i16: /array_value/0");

        UNIT_ASSERT_VALUES_EQUAL(42, conf.As<i16>("/int_value"));
    }

    Y_UNIT_TEST(createLog) {
        TConfig conf = TConfig::ReadFromMemory(
            R"({
                "log": {
                    "file": "config_temp_log_file",
                    "level": "INFO",
                    "print_level": true
                }
            })");

        std::unique_ptr<TFileLogger> log = conf.CreateLogger("/log");
        UNIT_ASSERT(log.get());
        UNIT_ASSERT_VALUES_EQUAL((int)ILogger::ELevel::INFO, (int)log->GetLevel());
        log->Debug("debug: %s", "ON");
        log->Info("info: %d %s", 2, "test");
        log->Warning("warning!");
        log->Error("error: %o %x %X %#x", 11, 12, 13, 14);
        log.reset();

        {
            TFileInput log_file("config_temp_log_file");
            UNIT_ASSERT_VALUES_EQUAL("INFO: info: 2 test\n"
                                     "WARNING: warning!\n"
                                     "ERROR: error: 13 c D 0xe\n",
                                     log_file.ReadAll());
        }

        conf.InitCommonLog("/log");
        TLog::Warning("common warning");
        TDbPoolLog().Error("misterious dbpool error!");
        TLog::Reset();

        {
            TFileInput log_file("config_temp_log_file");
            UNIT_ASSERT_VALUES_EQUAL("INFO: info: 2 test\n"
                                     "WARNING: warning!\n"
                                     "ERROR: error: 13 c D 0xe\n"
                                     "WARNING: common warning\n",
                                     log_file.ReadAll());
        }
    }

    Y_UNIT_TEST(getKeyFromPath) {
        UNIT_ASSERT_VALUES_EQUAL("", TConfig::GetKeyFromPath(""));

        UNIT_ASSERT_VALUES_EQUAL("bar", TConfig::GetKeyFromPath("/foo/bar"));
        UNIT_ASSERT_VALUES_EQUAL("foo", TConfig::GetKeyFromPath("/foo"));
        UNIT_ASSERT_VALUES_EQUAL("", TConfig::GetKeyFromPath("/foo/bar/"));
    }

    Y_UNIT_TEST(getVectors) {
        const TConfig conf = TConfig::ReadFromMemory(
            R"({
                "string_values": ["some_str", "some_str2", "1"],
                "empty_string_values": [""],
                "empty_array_values": [],
                "int_values": [-100500, -1, 123, 120391],
                "uint_values": [0, 1, 43],
                "double_values": [0.25, -123.321],

                "mixed_array": ["some_str", 1]
            })");

        UNIT_ASSERT_VALUES_EQUAL((std::vector<TString>{"some_str", "some_str2", "1"}), conf.As<std::vector<TString>>("/string_values"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<TString>>("/empty_string_values"), TConfig::TEmptyException, "value is empty: /empty_string_values/0");
        UNIT_ASSERT_VALUES_EQUAL(0, conf.As<std::vector<i32>>("/empty_array_values").size());
        UNIT_ASSERT_VALUES_EQUAL(0, conf.As<std::vector<ui32>>("/empty_array_values").size());
        UNIT_ASSERT_VALUES_EQUAL(0, conf.As<std::vector<i64>>("/empty_array_values").size());
        UNIT_ASSERT_VALUES_EQUAL(0, conf.As<std::vector<ui64>>("/empty_array_values").size());
        UNIT_ASSERT_VALUES_EQUAL(0, conf.As<std::vector<double>>("/empty_array_values").size());
        UNIT_ASSERT_VALUES_EQUAL(0, conf.As<std::vector<TString>>("/empty_array_values").size());

        UNIT_ASSERT_VALUES_EQUAL((std::vector<double>{-100500, -1, 123, 120391}), conf.As<std::vector<double>>("/int_values"));
        UNIT_ASSERT_VALUES_EQUAL((std::vector<i32>{-100500, -1, 123, 120391}), conf.As<std::vector<i32>>("/int_values"));
        UNIT_ASSERT_VALUES_EQUAL((std::vector<i64>{-100500, -1, 123, 120391}), conf.As<std::vector<i64>>("/int_values"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<ui64>>("/int_values"), TConfig::TBadValueException, "jsonpoint is not ui64: /int_values/0");

        UNIT_ASSERT_VALUES_EQUAL((std::vector<i64>{0, 1, 43}), conf.As<std::vector<i64>>("/uint_values"));
        UNIT_ASSERT_VALUES_EQUAL((std::vector<double>{0, 1, 43}), conf.As<std::vector<double>>("/uint_values"));
        UNIT_ASSERT_VALUES_EQUAL((std::vector<ui32>{0, 1, 43}), conf.As<std::vector<ui32>>("/uint_values"));
        UNIT_ASSERT_VALUES_EQUAL((std::vector<ui64>{0, 1, 43}), conf.As<std::vector<ui64>>("/uint_values"));

        UNIT_ASSERT_VALUES_EQUAL((std::vector<double>{0.25, -123.321}), conf.As<std::vector<double>>("/double_values"));
        UNIT_ASSERT_VALUES_EQUAL((std::vector<double>{0.25, -123.321}), conf.As<std::vector<double>>("/double_values"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<ui64>>("/double_values"), TConfig::TBadValueException, "jsonpoint is not ui64: /double_values/0");

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<i32>>("/mixed_array"), TConfig::TBadValueException, "jsonpoint is not i32: /mixed_array/0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<ui32>>("/mixed_array"), TConfig::TBadValueException, "jsonpoint is not ui32: /mixed_array/0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<i64>>("/mixed_array"), TConfig::TBadValueException, "jsonpoint is not i64: /mixed_array/0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<ui64>>("/mixed_array"), TConfig::TBadValueException, "jsonpoint is not ui64: /mixed_array/0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<double>>("/mixed_array"), TConfig::TBadValueException, "jsonpoint is not double: /mixed_array/0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<std::vector<TString>>("/mixed_array"), TConfig::TBadValueException, "jsonpoint is not string: /mixed_array/1");
    }
}
