#include <passport/infra/libs/cpp/json/writer.h>

#include <passport/infra/libs/cpp/utils/string/format.h>

#include <contrib/libs/rapidjson/include/rapidjson/document.h>
#include <contrib/libs/rapidjson/include/rapidjson/writer.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NJson;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspJsonCommon) {
    Y_UNIT_TEST(jsonRead) {
        rapidjson::StringBuffer sb;
        TRapidWriter wr(sb);
        rapidjson::GenericDocument<rapidjson::ASCII<>> docAscii;
        rapidjson::Document doc; // utf8
        TString buf;

        UNIT_ASSERT(docAscii.Parse("").HasParseError());
        UNIT_ASSERT(docAscii.Parse("[").HasParseError());
        UNIT_ASSERT(docAscii.Parse("}").HasParseError());
        //    UNIT_ASSERT(docAscii.Parse("[\"\\uDE01\"]").HasParseError()); !!!!!!!!!!!!!!!!!!!!!!!!!

        UNIT_ASSERT(!doc.Parse("1234").HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.IsInt());
        UNIT_ASSERT_VALUES_EQUAL(1234, doc.GetInt());

        UNIT_ASSERT(!doc.Parse("[\"a\", true, 1234, null]").HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.IsArray());
        UNIT_ASSERT_VALUES_EQUAL(rapidjson::SizeType(4ul), doc.Size());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[0].IsString());
        UNIT_ASSERT_VALUES_EQUAL("a", TString(doc[0].GetString()));
        UNIT_ASSERT_VALUES_EQUAL(true, doc[1].IsBool());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[1].GetBool());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[2].IsInt());
        UNIT_ASSERT_VALUES_EQUAL(1234, doc[2].GetInt());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[3].IsNull());

        doc.Accept(wr);
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("[\"a\",true,1234,null]"), buf);
        UNIT_ASSERT(!doc.Parse(buf.c_str()).HasParseError());

        UNIT_ASSERT(!doc.Parse("{ \"a\"  : \"b\", \"c\" : false, \"d\" : null }").HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.IsObject());
        UNIT_ASSERT_VALUES_EQUAL(rapidjson::SizeType(3ul), doc.MemberCount());
        UNIT_ASSERT(doc.HasMember("a"));
        UNIT_ASSERT_VALUES_EQUAL(true, doc["a"].IsString());
        UNIT_ASSERT_VALUES_EQUAL("b", TString(doc["a"].GetString()));
        UNIT_ASSERT(doc.HasMember("c"));
        UNIT_ASSERT_VALUES_EQUAL(true, doc["c"].IsBool());
        UNIT_ASSERT_VALUES_EQUAL(false, doc["c"].GetBool());
        UNIT_ASSERT(doc.HasMember("d"));
        UNIT_ASSERT_VALUES_EQUAL(true, doc["d"].IsNull());

        sb.Clear();
        wr.Reset(sb);
        doc.Accept(wr);
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("{\"a\":\"b\",\"c\":false,\"d\":null}"), buf);
        UNIT_ASSERT(!doc.Parse(buf.c_str()).HasParseError());

        TString fieldStr("{ \"поле\" : \"\tРусское Поле\n\", \"\" : [ 1, null, 1]}");
        UNIT_ASSERT(doc.Parse(fieldStr.c_str()).HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(static_cast<int>(rapidjson::kParseErrorStringInvalidEncoding),
                                 static_cast<int>(doc.GetParseError()));
        fieldStr.assign(NUtils::EscapeEol(fieldStr)); // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! todo http://www.ietf.org/rfc/rfc4627

        UNIT_ASSERT(!doc.Parse(fieldStr.c_str()).HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.IsObject());
        UNIT_ASSERT_VALUES_EQUAL(rapidjson::SizeType(2ul), doc.MemberCount());
        UNIT_ASSERT(doc.HasMember("поле"));
        UNIT_ASSERT_VALUES_EQUAL(true, doc["поле"].IsString());
        UNIT_ASSERT_VALUES_EQUAL("\tРусское Поле\n", TString(doc["поле"].GetString()));
        UNIT_ASSERT(doc.HasMember(""));
        UNIT_ASSERT_VALUES_EQUAL(true, doc[""].IsArray());
        UNIT_ASSERT_VALUES_EQUAL(rapidjson::SizeType(3ul), doc[""].Size());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[""][0].IsInt());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[""][1].IsNull());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[""][2].IsInt());

        sb.Clear();
        wr.Reset(sb);
        doc.Accept(wr);
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("{\"\\u043F\\u043E\\u043B\\u0435\":\"\\t\\u0420\\u0443\\u0441\\u0441\\u043A\\u043E\\u0435 \\u041F\\u043E\\u043B\\u0435\\n\",\"\":[1,null,1]}"), buf);
        UNIT_ASSERT(!doc.Parse(buf.c_str()).HasParseError());

        UNIT_ASSERT(!doc.Parse("{ \"\xd0\xbe_\xd0\x9e\" : \"\\u041E_\\u043E\\\\\"}").HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.IsObject());
        UNIT_ASSERT_VALUES_EQUAL(rapidjson::SizeType(1ul), doc.MemberCount());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.HasMember("о_О"));
        UNIT_ASSERT_VALUES_EQUAL(true, doc["о_О"].IsString());
        UNIT_ASSERT_VALUES_EQUAL("О_о\\", TString(doc["о_О"].GetString()));

        sb.Clear();
        wr.Reset(sb);
        doc.Accept(wr);
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("{\"\\u043E_\\u041E\":\"\\u041E_\\u043E\\\\\"}"), buf);
        UNIT_ASSERT(!doc.Parse(buf.c_str()).HasParseError());

        TString smileStr("[\"\xf0\x9f\x98\x82\", \"+\\u041E\\uD83D\\uDE02+\", \"\u041E\u043e\n\" ]"); // String is mostly escaped already
        TString smileStrEsc("[\"\xf0\x9f\x98\x82\", \"+\\u041E\\uD83D\\uDE02+\", \"\u041E\u043e\\n\" ]");

        UNIT_ASSERT(doc.Parse(smileStr.c_str()).HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(static_cast<int>(rapidjson::kParseErrorStringInvalidEncoding),
                                 static_cast<int>(doc.GetParseError()));

        UNIT_ASSERT(!doc.Parse(smileStrEsc.c_str()).HasParseError());
        UNIT_ASSERT_VALUES_EQUAL(true, doc.IsArray());
        UNIT_ASSERT_VALUES_EQUAL(rapidjson::SizeType(3ul), doc.Size());
        UNIT_ASSERT_VALUES_EQUAL(true, doc[0].IsString());
        UNIT_ASSERT_VALUES_EQUAL("😂", TString(doc[0].GetString()));
        UNIT_ASSERT_VALUES_EQUAL(true, doc[1].IsString());
        UNIT_ASSERT_VALUES_EQUAL("+О😂+", TString(doc[1].GetString()));
        UNIT_ASSERT_VALUES_EQUAL(true, doc[2].IsString());
        UNIT_ASSERT_VALUES_EQUAL("Оо\n", TString(doc[2].GetString()));

        sb.Clear();
        wr.Reset(sb);
        doc.Accept(wr);
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("[\"\\uD83D\\uDE02\",\"+\\u041E\\uD83D\\uDE02+\",\"\\u041E\\u043E\\n\"]"), buf);
        UNIT_ASSERT(!doc.Parse(buf.c_str()).HasParseError());
    }

    Y_UNIT_TEST(jsonWrite) {
        rapidjson::StringBuffer sb;
        TRapidWriter wr(sb);
        TString buf;

        //{"1":["a", true, 1234, null],
        // "2":{ "a"  : "b", "c" : false, "d" : null }}
        wr.StartObject();
        wr.String("1");
        {
            wr.StartArray();
            wr.String("a");
            wr.Bool(true);
            wr.Int(1234);
            wr.Null();
            wr.EndArray();
        }
        wr.String("2");
        {
            wr.StartObject();
            wr.String("a");
            wr.String("b");
            wr.String("c");
            wr.Bool(false);
            wr.String("d");
            wr.Null();
            wr.EndObject();
        }
        wr.EndObject();
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("{\"1\":[\"a\",true,1234,null],"
                                         "\"2\":{\"a\":\"b\",\"c\":false,\"d\":null}}"),
                                 buf);

        // {"3":{ "поле" : "\tРусское Поле\n", "" : [ 1, null, 1]}}
        sb.Clear();
        wr.Reset(sb);
        wr.StartObject();
        wr.String("3");
        {
            wr.StartObject();
            wr.String("поле");
            wr.String("\tРусское Поле\n");
            wr.String("");
            wr.StartArray();
            wr.Int(1);
            wr.Null();
            wr.Int(1);
            wr.EndArray();
            wr.EndObject();
        }
        wr.EndObject();
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("{\"3\":{\"\\u043F\\u043E\\u043B\\u0435\":\"\\t\\u0420\\u0443"
                                         "\\u0441\\u0441\\u043A\\u043E\\u0435 "
                                         "\\u041F\\u043E\\u043B\\u0435\\n\",\"\":[1,null,1]}}"),
                                 buf);

        // {"4":["😂","+О😂+","Оо\n"]}
        sb.Clear();
        wr.Reset(sb);
        wr.StartObject();
        wr.String("4");
        {
            wr.StartArray();
            wr.String("😂");
            wr.String("+О😂+");
            wr.String("Оо\n");
            wr.EndArray();
        }
        wr.EndObject();
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString("{\"4\":[\"\\uD83D\\uDE02\","
                                         "\"+\\u041E\\uD83D\\uDE02+\","
                                         "\"\\u041E\\u043E\\n\"]}"),
                                 buf);

        // {"key":"/"]}
        sb.Clear();
        wr.Reset(sb);
        wr.StartObject();
        wr.String("key");
        wr.String("/");
        wr.EndObject();
        buf.assign(sb.GetString());
        UNIT_ASSERT_VALUES_EQUAL(TString(R"({"key":"\/"})"), buf);
    }
}
