#include <passport/infra/libs/cpp/json/schema.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NJson;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspJsonSchema) {
    Y_UNIT_TEST(invalidSchema) {
        TString schema =
            R"({
                "required": ["some_key_1", "some_key_2"]
            )";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TJsonSchema(schema), yexception, "schema must be valid json object");

        schema.push_back('}');
        UNIT_ASSERT_NO_EXCEPTION(TJsonSchema(schema));
    }

    static const TString SCHEMA =
        R"({
          "type": "object",
          "properties": {
            "some_key_1": {
              "type": "integer"
            },
            "some_key_2": {
              "type": "object",
              "properties": {
                "some_key_3": {
                  "type": "array",
                  "items": [{
                    "type": "object",
                    "properties": {
                      "some_key_4": {
                        "type": "string",
                        "enum": ["active", "not_active", "used"]
                      },
                      "some_key_5": {
                        "type": "integer"
                      },
                      "some_key_6": {
                        "type": "boolean"
                      }
                    },
                    "required": ["some_key_4", "some_key_5"]
                  }]
                }
              }
            }
          },
          "required": ["some_key_1", "some_key_2"]
        })";

    static rapidjson::Document MakeDocument(const TString& json) {
        rapidjson::Document d;
        Y_ENSURE(TReader::DocumentAsObject(json, d));
        return d;
    }

    Y_UNIT_TEST(validateJsonSchemaRequired) {
        const TJsonSchema s(SCHEMA);

        {
            TString json = "{}";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #. keyword: required. input: #", err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117
                })";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #. keyword: required. input: #", err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                    }
                })";
            TString err;
            UNIT_ASSERT(s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("", err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                        ]
                    }
                })";
            TString err;
            UNIT_ASSERT(s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("", err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "active"
                            }
                        ]
                    }
                })";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #/properties/some_key_2/properties/some_key_3/items/0. keyword: required. input: #/some_key_2/some_key_3/0",
                                     err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "active",
                                "some_key_5" : 129
                            }
                        ]
                    }
                })";
            TString err;
            UNIT_ASSERT(s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("", err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "active",
                                "some_key_5" : 129,
                                "some_key_6" : false
                            }
                        ]
                    },
                    "unspecified field" : null
                })";
            TString err;
            UNIT_ASSERT(s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("", err);
        }
    }

    Y_UNIT_TEST(validateJsonSchemaType) {
        const TJsonSchema s(SCHEMA);

        {
            TString json =
                R"({
                    "some_key_1" : false,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "active",
                                "some_key_5" : 129,
                                "some_key_6" : false
                            }
                        ]
                    },
                    "unspecified field" : null
                })";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #/properties/some_key_1. keyword: type. input: #/some_key_1",
                                     err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : [{
                        "some_key_3" : [
                            {
                                "some_key_4" : "active",
                                "some_key_5" : 129,
                                "some_key_6" : false
                            }
                        ]
                    }],
                    "unspecified field" : null
                })";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #/properties/some_key_2. keyword: type. input: #/some_key_2",
                                     err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "adasdas",
                                "some_key_5" : 129,
                                "some_key_6" : false
                            }
                        ]
                    },
                    "unspecified field" : null
                })";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #/properties/some_key_2/properties/some_key_3/items/0/properties/some_key_4. keyword: enum. input: #/some_key_2/some_key_3/0/some_key_4",
                                     err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "active",
                                "some_key_5" : 129,
                                "some_key_6" : 0
                            }
                        ]
                    },
                    "unspecified field" : null
                })";
            TString err;
            UNIT_ASSERT(!s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("schema: #/properties/some_key_2/properties/some_key_3/items/0/properties/some_key_6. keyword: type. input: #/some_key_2/some_key_3/0/some_key_6",
                                     err);
        }

        {
            TString json =
                R"({
                    "some_key_1" : 117,
                    "some_key_2" : {
                        "some_key_3" : [
                            {
                                "some_key_4" : "active",
                                "some_key_5" : 129,
                                "some_key_6" : false
                            }
                        ]
                    },
                    "unspecified field" : "omg"
                })";
            TString err;
            UNIT_ASSERT(s.Validate(MakeDocument(json), &err));
            UNIT_ASSERT_VALUES_EQUAL("", err);
        }
    }
}
