#include "writer.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NJson {
    static TStringBuf::const_iterator FindNonUtf8Char(const TStringBuf val) {
        TStringBuf::const_iterator it = val.begin();

        while ((it = std::find_if(it, val.end(), [](const unsigned char c) {
                    return c > 0x7f;
                })) != val.end()) {
            const size_t len = NUtils::Utf8CharLength(it, val.end());
            if (len == 0) {
                break;
            }
            it += len;
        }
        return it;
    }

    static TString SkipNonUtf8Chars(const TStringBuf val, TStringBuf::const_iterator it) {
        TString result;
        result.reserve(val.size());
        result.assign(val.begin(), it);

        while (it != val.end()) {
            const size_t len = NUtils::Utf8CharLength(it, val.end());

            if (len == 0) {
                // Skip suspicious chars
                ++it;
                continue;
            }

            for (size_t idx = 0; idx < len; ++idx) {
                result.push_back(*it++);
            }
        }

        return result;
    }

    static void LogJsonWarning(const TStringBuf val) {
        TLog::Warning() << "Invalid UTF symbols on attempt to serialize json. String(hex): " << NUtils::Bin2hex(val);
    }

    bool TRapidWriter::WriteKeyChecked(const TStringBuf name) {
        if (!TRapidWriterBase::Key(name.data(), name.size())) {
            TLog::Error() << "Failed to serialize json key. Key(hex): " << NUtils::Bin2hex(name);
            return false;
        }
        return true;
    }

    bool TRapidWriter::Key(const TStringBuf name) {
        auto it = FindNonUtf8Char(name);

        if (it != name.end()) { // has non-utf8, need escaping
            LogJsonWarning(name);
            TString escaped = SkipNonUtf8Chars(name, it);
            return WriteKeyChecked(escaped);
        }
        return WriteKeyChecked(name);
    }

    bool TRapidWriter::WriteStringChecked(const TStringBuf val) {
        if (!TRapidWriterBase::String(val.data(), val.size())) {
            TLog::Error() << "Failed to serialize json string. String(hex): " << NUtils::Bin2hex(val);
            return false;
        }
        return true;
    }

    bool TRapidWriter::String(const TStringBuf val) {
        auto it = FindNonUtf8Char(val);

        if (it != val.end()) {
            LogJsonWarning(val);
            TString escaped = SkipNonUtf8Chars(val, it);
            return WriteStringChecked(escaped);
        }
        return WriteStringChecked(val);
    }
}
