#pragma once

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <contrib/libs/rapidjson/include/rapidjson/document.h>
#include <contrib/libs/rapidjson/include/rapidjson/writer.h>

#include <util/generic/noncopyable.h>
#include <util/generic/string.h>

#include <memory>

namespace NPassport::NJson {
    using TRapidWriterBase = rapidjson::Writer<rapidjson::StringBuffer, rapidjson::UTF8<>, rapidjson::ASCII<>>;

    class TRapidWriter: public TRapidWriterBase {
    public:
        using TRapidWriterBase::Key;
        using TRapidWriterBase::String;
        using TRapidWriterBase::TRapidWriterBase;

        bool Key(const TStringBuf name);
        bool String(const TStringBuf val);

    private:
        bool WriteKeyChecked(const TStringBuf name);
        bool WriteStringChecked(const TStringBuf val);
    };

    class TArray;
    class TObject;

    class TWriter: TMoveOnly {
    public:
        TWriter(TString& buf)
            : Writer_(Sb_)
            , Buf_(buf)
        {
        }

        ~TWriter() {
            try {
                Buf_.assign(Sb_.GetString(), Sb_.GetSize());
            } catch (...) {
                Y_FAIL("%s", CurrentExceptionMessage().c_str());
            }
        }

    private:
        rapidjson::StringBuffer Sb_;
        TRapidWriter Writer_;
        TString& Buf_;

        friend class TArray;
        friend class TObject;
    };

    class TObject: TMoveOnly {
    public:
        TObject(TWriter& writer)
            : Writer_(writer.Writer_)
        {
            Writer_.StartObject();
        }

        TObject(TArray& writer);

        TObject(TObject& writer, const TStringBuf name)
            : Writer_(writer.Writer_)
        {
            Writer_.Key(name);
            Writer_.StartObject();
        }

        ~TObject() {
            Writer_.EndObject();
        }

        // basic types
        void Add(const TStringBuf key, int val) {
            Writer_.Key(key);
            Writer_.Int(val);
        }

        void Add(const TStringBuf key, i64 val) {
            Writer_.Key(key);
            Writer_.Int64(val);
        }

        void Add(const TStringBuf key, unsigned int val) {
            Writer_.Key(key);
            Writer_.Uint(val);
        }

        void Add(const TStringBuf key, ui64 val) {
            Writer_.Key(key);
            Writer_.Uint64(val);
        }

        void Add(const TStringBuf key, double val) {
            Writer_.Key(key);
            Writer_.Double(val);
        }

        void Add(const TStringBuf key, bool val) {
            Writer_.Key(key);
            Writer_.Bool(val);
        }

        void Add(const TStringBuf key, const TStringBuf val) {
            Writer_.Key(key);
            Writer_.String(val);
        }

        void Add(const TStringBuf key, const char* val) {
            Writer_.Key(key);
            if (val) {
                Writer_.String(TStringBuf(val));
            } else {
                Writer_.Null();
            }
        }

        void Add(const TStringBuf key, const rapidjson::Document& val) {
            Writer_.Key(key);
            val.Accept(Writer_);
        }

    private:
        TRapidWriter& Writer_;
        friend class TArray;
    };

    class TArray: TMoveOnly {
    public:
        TArray(TWriter& writer)
            : Writer_(writer.Writer_)
        {
            Writer_.StartArray();
        }

        TArray(TArray& writer)
            : Writer_(writer.Writer_)
        {
            Writer_.StartArray();
        }

        TArray(TObject& writer, const TStringBuf name)
            : Writer_(writer.Writer_)
        {
            Writer_.Key(name);
            Writer_.StartArray();
        }

        ~TArray() {
            Writer_.EndArray();
        }

        // basic types
        void Add(int val) {
            Writer_.Int(val);
        }

        void Add(i64 val) {
            Writer_.Int64(val);
        }

        void Add(unsigned int val) {
            Writer_.Uint(val);
        }

        void Add(ui64 val) {
            Writer_.Uint64(val);
        }

        void Add(double val) {
            Writer_.Double(val);
        }

        void Add(bool val) {
            Writer_.Bool(val);
        }

        void Add(const TStringBuf str) {
            Writer_.String(str);
        }

        void Add(const char* val) {
            if (val) {
                Writer_.String(TStringBuf(val));
            } else {
                Writer_.Null();
            }
        }

    private:
        TRapidWriter& Writer_;
        friend class TObject;
    };

    inline TObject::TObject(TArray& writer)
        : Writer_(writer.Writer_)
    {
        Writer_.StartObject();
    }
}
