#include "keys.h"

#include <contrib/restricted/keyutils/keyutils.h>

#include <util/system/error.h>

namespace NPassport::NKeyutils {
#if defined(_linux_)
    static const TString KEY_TYPE = "user";

    static long GetPersistentKeyring() {
        const uid_t currentUid = -1;
        return keyctl_get_persistent(currentUid, KEY_SPEC_PROCESS_KEYRING);
    }

    TString TReader::Read(const TString& name) {
        auto throwError = [&](TStringBuf description) {
            ythrow yexception()
                << "Keyutils: read failed for key '" << name << "': " << description
                << ". " << LastSystemErrorText();
        };

        const long persistent = GetPersistentKeyring();
        if (-1 == persistent) {
            throwError("failed to get persistent keyring");
        }
        const key_serial_t key = request_key(KEY_TYPE.c_str(), name.c_str(), nullptr, persistent);
        if (key == -1) {
            throwError("key was not found");
        }

        long size = keyctl_read(key, nullptr, 0);
        if (size == 0) {
            throwError("value is empty");
        }
        if (size < 0) {
            throwError("failed to read key value");
        }

        TString res(size, 0);
        long newSize = keyctl_read(key, (char*)res.data(), res.size());
        if (newSize != size) {
            throwError("race was detected: key size changed in tryReadKey()");
        }

        return res;
    }

    void TWriter::Write(const TString& name, const TStringBuf value) {
        auto throwError = [&](TStringBuf description) {
            ythrow yexception()
                << "Keyutils: write failed for key '" << name << "':" << description
                << ". " << LastSystemErrorText();
        };

        const long persistent = GetPersistentKeyring();
        if (-1 == persistent) {
            throwError("failed to get persistent keyring");
        }

        const key_serial_t key = add_key(
            KEY_TYPE.c_str(),
            name.c_str(),
            value.data(),
            value.size(),
            persistent);
        if (-1 == key) {
            throwError("failed to add");
        }

        // allow to interact only for current user
        if (-1 == keyctl_setperm(key, KEY_USR_ALL)) {
            throwError("failed to setperm");
        }
    }
#else
    TString TReader::Read(const TString&) {
        ythrow yexception() << "not implemented";
    }

    void TWriter::Write(const TString&, const TStringBuf) {
        ythrow yexception() << "not implemented";
    }
#endif
}
