#include "base_parser.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>

namespace NPassport::NLb {
    void TBaseParser::ReserveForTopic(const TTopicData& data) {
        Reserve(1024 * data.Data.size());
    }

    void TBaseParser::ParseRows(const TData& data) {
        size_t size = 0;
        for (const NLb::TTopicData& d : data.Messages) {
            size += 1024 * d.Data.size();
        }

        Reserve(size);

        for (const NLb::TTopicData& d : data.Messages) {
            ParseRows(d);
        }
    }

    void TBaseParser::ParseRows(const TTopicData& data) {
        for (const TChunk& message : data.Data) {
            TStringBuf s(message.Data);
            while (s) {
                TStringBuf buf;
                TStringBuf restPart;

                try {
                    if (!s.TrySplit('\n', buf, restPart)) {
                        buf = s;
                        ythrow yexception() << "row is not full";
                    }

                    // Fix from PASSP-32663 (updated due to PASSP-34193).
                    // We are essentially turning invalid lines into valid
                    // by removing leading end-of-string character
                    buf = SkipZeroPrefix(buf);

                    if (buf && ParseRow(buf)) {
                        ++TotalRowsIn_;
                    }
                } catch (const std::exception& e) {
                    ++Errors_;
                    LogBadLine(buf);
                    TLog::Warning() << ParserName_ << ": Failed to parse row '"
                                    << buf << "': " << e.what();
                }

                s = restPart;
            }
        }
    }

    TStringBuf TBaseParser::SkipZeroPrefix(TStringBuf buf) {
        ui64 zeroPrefixLength = 0;
        while (buf.SkipPrefix(TStringBuf("\x00", 1))) {
            ++zeroPrefixLength;
        }

        if (zeroPrefixLength > 0) {
            ++Errors_;
            TLog::Warning() << ParserName_ << ": Skipped " << zeroPrefixLength
                            << " leading end-of-string characters";
        }

        return buf;
    }

    void TParserWithBadLineLog::LogBadLine(TStringBuf line) {
        if (!Logger_) {
            return;
        }

        Logger_->Log(NUtils::CreateStr(line, "\n"));
    }
}
