#pragma once

#include "bad_line_loggers.h"
#include "base_parser.h"

#include <passport/infra/libs/cpp/unistat/diff.h>

#include <array>

namespace NPassport::NLb {
    template <class LogTypeTraits>
    class TBaseProcessor {
        using TLogType = typename LogTypeTraits::TType;

        static_assert(std::is_enum_v<TLogType>, "");

    public:
        TBaseProcessor(TBadLineLoggers<LogTypeTraits> badLinesLoggers);

    protected:
        void AddUnistat(NUnistat::TBuilder& builder) const;

        void CountUnistatSignals(TLogType type, const NLb::TBaseParser& parser);

    protected:
        TBadLineLoggers<LogTypeTraits> BadLinesLoggers_;

    private:
        using TUnistatSignalsForLog = std::array<std::unique_ptr<NUnistat::TSignalDiff<>>,
                                                 LogTypeTraits::count_>;

        static TUnistatSignalsForLog CreateArray(TString key) {
            TUnistatSignalsForLog res;

            for (size_t idx = 0; idx < res.size(); ++idx) {
                res[idx] = std::make_unique<NUnistat::TSignalDiff<>>(
                    "lb.proc." + key + "." + LogTypeTraits::ToString((TLogType)idx));
            }

            return res;
        }

    private:
        NUnistat::TSignalDiff<> UnistatRowsIn_ = {"lb.proc.rows"};
        NUnistat::TSignalDiff<> UnistatErrorsIn_ = {"lb.proc.errors"};
        TUnistatSignalsForLog UnistatRows_;
        TUnistatSignalsForLog UnistatErrors_;
    };

    template <typename LogTypeTraits>
    TBaseProcessor<LogTypeTraits>::TBaseProcessor(TBadLineLoggers<LogTypeTraits> badLinesLoggers)
        : BadLinesLoggers_(std::move(badLinesLoggers))
        , UnistatRows_(CreateArray("rows"))
        , UnistatErrors_(CreateArray("errors"))
    {
    }

    template <typename LogTypeTraits>
    void TBaseProcessor<LogTypeTraits>::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(UnistatRowsIn_);
        builder.Add(UnistatErrorsIn_);

        auto add = [&builder](const TUnistatSignalsForLog& arr) {
            for (const std::unique_ptr<NUnistat::TSignalDiff<>>& u : arr) {
                builder.Add(*u);
            }
        };
        add(UnistatRows_);
        add(UnistatErrors_);
    }

    template <typename LogTypeTraits>
    void TBaseProcessor<LogTypeTraits>::CountUnistatSignals(TLogType type,
                                                            const TBaseParser& parser) {
        UnistatRowsIn_ += parser.GetRowsIn();
        UnistatErrorsIn_ += parser.GetErrorCount();

        size_t idx = (size_t)type;
        (*UnistatRows_[idx]) += parser.GetRowsIn();
        (*UnistatErrors_[idx]) += parser.GetErrorCount();
    }
}
