#pragma once

#include <passport/infra/libs/cpp/logbroker/reader/reader.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/unistat/busy_holder.h>

#include <util/stream/str.h>
#include <util/system/thread.h>
#include <util/thread/lfqueue.h>

#include <atomic>
#include <memory>

namespace NPassport::NLb {
    struct TReaderPoolSettings {
        size_t QueueSizeLimit = 32;
        size_t Workers = 8;
        size_t JoinMessages = 1;
    };

    template <typename LogType = int>
    struct TDataWithType {
        TData Data;
        LogType Type = {};
    };

    template <typename LogType = int>
    using TDataSet = TStackVec<TDataWithType<LogType>, 16>;

    template <typename DataType, typename LogType = int>
    class TReaderPool {
    public:
        struct TReader {
            NLb::TReaderThreadPtr Reader;
            LogType Type;
        };

        struct TRawData {
            TData Data;
            LogType Type;
            NThreading::TPromise<void> Promise;
        };
        using TRawDataPtr = std::shared_ptr<TRawData>;

        struct TParsedData {
            DataType Data;
            TStackVec<NThreading::TPromise<void>, 16> Promises;
            TTraceids Traceids;
        };
        using TParsedDataPtr = std::shared_ptr<TParsedData>;

    public:
        TReaderPool(const TReaderPoolSettings& settings)
            : Settings_(settings)
        {
            WorkersConfigCount_ = Settings_.Workers;
            Y_ENSURE(Settings_.JoinMessages, "value cannot be 0");
        }

        ~TReaderPool() {
            Stopping_ = true;

            for (TReader& thr : Readers_) {
                thr.Reader->StopLoop();
            }
            for (std::thread& thr : Workers_) {
                thr.join();
            }

            Workers_.clear();
            Readers_.clear();
        }

        size_t GetReadersCount() const {
            return Readers_.size();
        }

        void AddUnistat(NUnistat::TBuilder& builder) const {
            for (const TReader& thr : Readers_) {
                thr.Reader->AddUnistat(builder);
            }
            builder.Add(QueueInSize_);
            builder.Add(QueueOutSize_);
            builder.Add(BusyWorkersCount_);
            builder.Add(WorkersConfigCount_);
            builder.AddRow("lb.queue.size_limit_avvv", Settings_.QueueSizeLimit);
            builder.Add(Repsonses_);
            builder.Add(Datasets_);
        }

        TString GetErrorInfo() const {
            TStringStream s;

            for (const TReader& l : Readers_) {
                if (!l.Reader->IsOk()) {
                    s << "Lb is bad '" << l.Reader->GetServer() << "'" << Endl;
                }
            }

            return s.Str();
        }

        void AddReader(const TReaderSettings& settings, LogType type = {}) {
            Readers_.push_back(TReader{
                std::make_unique<NLb::TReaderThread>(settings),
                type,
            });
        }

        using TParser = std::function<DataType(const NLb::TDataSet<LogType>&)>;
        void Start(const TString& threadName, TParser parser) {
            StartWorkers(parser, threadName);
            StartReaders(threadName);
        }

        TParsedDataPtr TryGet() {
            TParsedDataPtr res;

            if (QueueOut_.Dequeue(&res)) {
                --QueueOutSize_;
            }

            return res;
        }

    protected: // fot tests
        void StartWorkers(TParser parser, const TString& threadName) {
            auto func = [this, parser, threadName]() -> void {
                thread_local TTheadNamer t(threadName + "lb_parse");

                while (!Stopping_.load(std::memory_order_relaxed)) {
                    try {
                        if (!Worker(parser)) {
                            Sleep(TDuration::Seconds(1));
                        }
                    } catch (const std::exception& e) {
                        TLog::Error() << "Lb: TReaderPool: exception in worker: " << e.what();
                    }
                }
            };

            for (size_t idx = 0; idx < Settings_.Workers; ++idx) {
                Workers_.push_back(std::thread(func));
            }
        }

        bool Worker(const TParser& parser) {
            if (QueueOutSize_.GetValue() >= Settings_.QueueSizeLimit) {
                return false;
            }

            TSmallVec<TRawDataPtr> raws;
            raws.reserve(Settings_.JoinMessages);

            {
                TRawDataPtr p;
                while (raws.size() < Settings_.JoinMessages && QueueIn_.Dequeue(&p)) {
                    --QueueInSize_;
                    raws.push_back(p);
                }
            }
            if (raws.empty()) {
                return false;
            }

            ++QueueOutSize_;

            NUnistat::TBusyHolder busy(BusyWorkersCount_);

            Parse(parser, raws);
            return true;
        }

        void Parse(const TParser& parser, TSmallVec<TRawDataPtr>& raws) {
            TDataSet<LogType> dataSet;
            dataSet.reserve(raws.size());

            TParsedDataPtr parsed = std::make_shared<TParsedData>();
            parsed->Promises.reserve(raws.size());
            parsed->Traceids.reserve(raws.size());

            for (TRawDataPtr& p : raws) {
                TRawData& raw = *p;

                parsed->Promises.push_back(std::move(raw.Promise));
                parsed->Traceids.push_back(raw.Data.Traceid);

                dataSet.push_back({
                    .Data = std::move(raw.Data),
                    .Type = raw.Type,
                });
            }

            ++Datasets_;
            Repsonses_ += dataSet.size();
            parsed->Data = parser(dataSet);

            QueueOut_.Enqueue(std::move(parsed));
        }

        void StartReaders(const TString& threadName) {
            for (TReader& thr : Readers_) {
                thr.Reader->StartLoop([this, threadName, type = thr.Type](
                                          NLb::TData&& data,
                                          NThreading::TPromise<void> promise) -> void {
                    thread_local TTheadNamer t(threadName + "lb_loop");

                    TRawDataPtr raw = CreateRaw(std::move(data), std::move(promise), type);

                    while (QueueInSize_.GetValue() >= Settings_.QueueSizeLimit) {
                        Sleep(TDuration::Seconds(1));
                        if (Stopping_.load(std::memory_order_relaxed)) {
                            return;
                        }
                    }

                    ++QueueInSize_;
                    QueueIn_.Enqueue(std::move(raw));
                });
            }
        }

        static TRawDataPtr CreateRaw(NLb::TData&& data,
                                     NThreading::TPromise<void> promise,
                                     const LogType& type) {
            TRawDataPtr res = std::make_shared<TRawData>();

            res->Promise = std::move(promise);
            res->Data = std::move(data);
            res->Type = type;

            return res;
        }

    private:
        struct TTheadNamer {
            TTheadNamer(const TString& threadName) {
                TThread::SetCurrentThreadName(threadName.c_str());
            }
        };

    protected: // fot tests
        std::atomic_bool Stopping_ = false;
        const TReaderPoolSettings Settings_;

        TLockFreeQueue<TRawDataPtr> QueueIn_;
        TLockFreeQueue<TParsedDataPtr> QueueOut_;
        NUnistat::TSignalAbsolute<> QueueInSize_ = {"lb.queue.in"};
        NUnistat::TSignalAbsolute<> QueueOutSize_ = {"lb.queue.out"};
        NUnistat::TSignalAbsolute<> BusyWorkersCount_ = {"lb.workers.busy"};
        NUnistat::TSignalAbsolute<> WorkersConfigCount_ = {"lb.workers.config"};
        NUnistat::TSignalDiff<> Repsonses_ = {"lb.repsonses"};
        NUnistat::TSignalDiff<> Datasets_ = {"lb.datasets"};

        std::vector<TReader> Readers_;
        std::vector<std::thread> Workers_;
    };

    template <typename T, typename LogType = int>
    using TReaderPoolPtr = std::shared_ptr<TReaderPool<T, LogType>>;
}
