#pragma once

#include "base_worker.h"

#include <passport/infra/libs/cpp/logbroker/processing/reader_pool.h>

namespace NPassport::NLb::NPusher {
    struct TBaseSettings {
        TString SignalPrefix;
        TString ErrorSignalName;
    };

    template <typename Derived, typename DataType, typename LogType = int>
    class TBasePusher {
    public:
        using TLbReaderPool = TReaderPool<DataType, LogType>;
        using TLbReaderPoolPtr = TReaderPoolPtr<DataType, LogType>;

    public:
        TBasePusher(const TBaseSettings& settings, TLbReaderPoolPtr readerPool);

        template <class WorkerType, class WorkerSettings = TBaseWorkerSettings>
        void CreateWorkers(size_t workers, const WorkerSettings& settings = TBaseWorkerSettings());

        NUnistat::TSignalDiff<>& GetUnistatErrors();

        void Stop();

    protected:
        void AddUnistatBase(NUnistat::TBuilder& builder) const;

    public:
        NUnistat::TBusyHolder<> GetBusyHolder();

        typename TLbReaderPool::TParsedDataPtr TryGet() {
            return ReaderPool_->TryGet();
        }

    private:
        TLbReaderPoolPtr ReaderPool_;

        std::vector<std::unique_ptr<TBaseWorker<Derived>>> Workers_;
        NUnistat::TSignalAbsolute<> BusyWorkers_;
        NUnistat::TSignalAbsolute<> WorkersConfigCount_;
        NUnistat::TSignalDiff<> Errors_;
    };

    template <typename Derived, typename DataType, typename LogType>
    TBasePusher<Derived, DataType, LogType>::TBasePusher(const TBaseSettings& settings, TLbReaderPoolPtr readerPool)
        : ReaderPool_(readerPool)
        , BusyWorkers_(settings.SignalPrefix + ".workers.busy")
        , WorkersConfigCount_(settings.SignalPrefix + ".workers.config")
        , Errors_(settings.ErrorSignalName)
    {
        TLog::Debug() << "Pusher was created";
    }

    template <typename Derived, typename DataType, typename LogType>
    void TBasePusher<Derived, DataType, LogType>::Stop() {
        for (std::unique_ptr<TBaseWorker<Derived>>& w : Workers_) {
            w->TriggerStopping();
        }
        for (std::unique_ptr<TBaseWorker<Derived>>& w : Workers_) {
            w->Stop();
        }
        Workers_.clear();
        TLog::Debug() << "Pusher was destroyed";
    }

    template <typename Derived, typename DataType, typename LogType>
    NUnistat::TSignalDiff<>& TBasePusher<Derived, DataType, LogType>::GetUnistatErrors() {
        return Errors_;
    }

    template <typename Derived, typename DataType, typename LogType>
    template <class WorkerType, class WorkerSettings>
    void TBasePusher<Derived, DataType, LogType>::CreateWorkers(size_t workers, const WorkerSettings& settings) {
        WorkersConfigCount_ = workers;

        for (size_t idx = 0; idx < workers; ++idx) {
            try {
                Workers_.push_back(std::make_unique<WorkerType>(settings, idx, static_cast<Derived&>(*this)));
            } catch (const std::exception& e) {
                TLog::Error() << "Failed to create worker#" << idx
                              << ": " << e.what();
                throw;
            }
        }
    }

    template <typename Derived, typename DataType, typename LogType>
    void TBasePusher<Derived, DataType, LogType>::AddUnistatBase(NUnistat::TBuilder& builder) const {
        builder.Add(BusyWorkers_);
        builder.Add(WorkersConfigCount_);
        builder.Add(Errors_);
    }

    template <typename Derived, typename DataType, typename LogType>
    NUnistat::TBusyHolder<> TBasePusher<Derived, DataType, LogType>::GetBusyHolder() {
        return NUnistat::TBusyHolder(BusyWorkers_);
    }
}
