#pragma once

#include <passport/infra/libs/cpp/unistat/busy_holder.h>
#include <passport/infra/libs/cpp/unistat/time_stat.h>
#include <passport/infra/libs/cpp/utils/exponential_backoff.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/datetime/base.h>
#include <util/stream/format.h>
#include <util/system/thread.h>

#include <atomic>
#include <thread>

namespace NPassport::NLb::NPusher {
    struct TBaseWorkerSettings {
        TDuration MinBackoff = TDuration::Seconds(1);
        TDuration MaxBackoff = TDuration::Seconds(90);
        TString ThreadName = "lbc_pusher";
    };

    using TTimeStatPtr = std::shared_ptr<NUnistat::TTimeStat>;

    template <class DataPusher>
    class TBaseWorker {
    public:
        using TProc = std::function<bool(typename DataPusher::TLbReaderPool::TParsedData&)>;

        TBaseWorker(const TBaseWorkerSettings& settings, size_t idx, DataPusher& parent, TProc proc);
        virtual ~TBaseWorker() = default;

        void TriggerStopping();
        void Stop();

        DataPusher& GetParent() const {
            return Parent_;
        }

    protected:
        const int Idx_;

    private:
        void Run();

    private:
        DataPusher& Parent_;
        TProc Proc_;

        std::atomic_bool Stopping_ = {false};
        NUtils::TExponentialBackoff Backoff_;
        std::thread Thread_;
    };

    template <class DataPusher>
    TBaseWorker<DataPusher>::TBaseWorker(const TBaseWorkerSettings& settings,
                                         size_t idx,
                                         DataPusher& parent,
                                         TProc proc)
        : Idx_(idx)
        , Parent_(parent)
        , Proc_(proc)
        , Backoff_(settings.MinBackoff, settings.MaxBackoff, 2, 0.2)
    {
        Thread_ = std::thread([this, threadName = settings.ThreadName]() {
            TThread::SetCurrentThreadName(threadName.c_str());
            Run();
        });
        TLog::Debug() << "worker#: " << idx << " created";
    }

    template <class DataPusher>
    void TBaseWorker<DataPusher>::TriggerStopping() {
        Stopping_ = true;
    }

    template <class DataPusher>
    void TBaseWorker<DataPusher>::Stop() {
        Stopping_ = true;
        Backoff_.Interrupt();
        Thread_.join();
        TLog::Debug() << "PushWorker#" << Idx_ << " was destroyed";
    }

    template <class DataPusher>
    void TBaseWorker<DataPusher>::Run() {
        TLog::Debug() << "worker#: " << Idx_ << ": started";

        while (!Stopping_.load(std::memory_order_relaxed)) {
            typename DataPusher::TLbReaderPool::TParsedDataPtr data = Parent_.TryGet();
            if (!data) {
                Sleep(TDuration::MilliSeconds(200));
                continue;
            }

            NUnistat::TBusyHolder busy = Parent_.GetBusyHolder();

            bool success = false;
            while (!success && !Stopping_.load(std::memory_order_relaxed)) {
                const TInstant start = TInstant::Now();

                try {
                    if (Proc_(*data)) {
                        success = true;
                        Backoff_.Decrease();
                        continue;
                    }
                } catch (const std::exception& e) {
                    TLog::Warning() << "worker#" << Idx_
                                    << ". Pusher: exception: " << e.what()
                                    << ". Took: " << HumanReadable(TInstant::Now() - start);
                }

                ++Parent_.GetUnistatErrors();
                TLog::Warning() << "worker#" << Idx_
                                << ". Error in Pusher: sleep for " << Backoff_.GetCurrentValue();
                Backoff_.Sleep();

                Backoff_.Increase();
            }
        }
    }
}
