#include "fetcher.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/string/join.h>

namespace NPassport::NLb {
    TFetcher::TFetcher(NPersQueue::TPQLib& pq,
                       const NPersQueue::TConsumerSettings& pqSettings,
                       const TReaderSettings& settings,
                       TIntrusivePtr<NPersQueue::ILogger>& logger,
                       NUnistat::TTimeStat& timestat)
        : Settings_(&settings)
        , LastCommitTime_(TInstant::Now())
        , Timestat_(timestat)
    {
        Consumer_ = pq.CreateConsumer(pqSettings, logger);
        Y_ENSURE(Consumer_);

        NThreading::TFuture<NPersQueue::TConsumerCreateResponse> consumerStart = Consumer_->Start();
        Y_ENSURE(consumerStart.Wait(Settings_->ConnectionTimeout),
                 "Future on CreateConsumer: timeout");

        const auto& resp = consumerStart.GetValueSync().Response;
        Y_ENSURE(!resp.HasError(), resp.GetError().GetDescription());

        SendRequestForNextMsg();
        TLog::Debug() << "lb::Fetcher created: " << Settings_->UnistatSignalId;
    }

    TFetcher::~TFetcher() {
        try {
            Cleanup();
        } catch (const std::exception& e) {
            TLog::Error("LbReader exception at fetcher destructor. %s", e.what());
        }

        TLog::Debug() << "lb::Fetcher deleted: " << Settings_->UnistatSignalId;
    }

    void TFetcher::WaitForAWhile() {
        if (NextData_) {
            Sleep(TDuration::MilliSeconds(500));
        } else {
            Ev_.WaitT(TDuration::MilliSeconds(500));
        }
    }

    void TFetcher::ForceCommit() {
        ui64 dataSize = 0;

        while (!Messages_.empty() && Messages_.front().Future.HasValue()) {
            ToCommit_.push_back(Messages_.front().Cookie);
            Timestat_.Insert(TInstant::Now() - Messages_.front().StartTime);
            dataSize += Messages_.front().DataSize;
            Messages_.pop_front();
        }

        if (!ToCommit_.empty()) {
            TLog::Debug() << "Lb sending commit for " << Settings_->UnistatSignalId
                          << ": " << JoinRange(",", ToCommit_.begin(), ToCommit_.end());
            Consumer_->Commit(ToCommit_);
            ToCommit_.clear();
            Settings_->UncommittedMemory->Release(Settings_->UnistatSignalId, dataSize);
            LastCommitTime_ = TInstant::Now();
        }
    }

    ui64 TFetcher::Process(TReader::TLoopFunc func) {
        ui64 msgCount = ProcessReads(func);

        Commit();

        return msgCount;
    }

    ui64 TFetcher::ProcessReads(TReader::TLoopFunc func) {
        ui64 dataSize = 0;
        ui64 messages = 0;
        TString traceids;

        if (!NextData_) {
            NextData_ = TryGetNextData();
        }

        while (NextData_) {
            if (!Settings_->UncommittedMemory->TryAcquire(Settings_->UnistatSignalId,
                                                          NextData_->DataSize)) {
                break;
            }

            if (traceids) {
                traceids.push_back(',');
            }
            NUtils::Append(traceids, NextData_->Data.Traceid);

            dataSize += NextData_->DataSize;
            messages += NextData_->Messages;

            NThreading::TPromise<void> promise = NThreading::NewPromise<void>();
            NThreading::TFuture<void> future = promise.GetFuture();

            func(std::move(NextData_->Data), std::move(promise));

            Messages_.push_back(TMessage{
                .Future = std::move(future),
                .DataSize = NextData_->DataSize,
                .Cookie = NextData_->Cookie,
            });

            SendRequestForNextMsg();
            NextData_ = TryGetNextData();
        }

        if (traceids) {
            TLog::Debug() << "Lb got " << messages
                          << " messages (" << dataSize << " bytes)"
                          << " with: " << traceids;
        }

        return dataSize;
    }

    std::optional<TFetcher::TDataWithInfo> TFetcher::TryGetNextData() {
        while (Read_.HasValue()) {
            const NPersQueue::TReadResponse& resp = Read_.GetValue().Response;
            const NPersQueue::EMessageType& type = Read_.GetValue().Type;

            Y_ENSURE_EX(type != NPersQueue::EMT_ERROR,
                        yexception() << "Failed to Read (code=" << (int)resp.GetError().GetCode()
                                     << "): " << resp.GetError().GetDescription().c_str());

            Y_ENSURE_EX(type == NPersQueue::EMT_DATA || type == NPersQueue::EMT_COMMIT,
                        yexception() << "unknown response type " << (int)type << " msg " << resp);

            if (type == NPersQueue::EMT_DATA) {
                return TranslateResp(resp, NUtils::CreateStr(Settings_->UnistatSignalId,
                                                             ":",
                                                             resp.GetData().GetCookie()));
            }

            SendRequestForNextMsg();
        }

        return {};
    }

    void TFetcher::Commit() {
        if ((TInstant::Now() - LastCommitTime_) < Settings_->CommitPeriod) {
            return;
        }

        ForceCommit();
    }

    void TFetcher::SendRequestForNextMsg() {
        Read_ = Consumer_->GetNextMessage();
        Read_.Subscribe([event = Ev_](const auto&) mutable { event.Signal(); });
    }

    static const TString EXTRA_FIELD_FILE = "file";
    static const TString EXTRA_FIELD_SERVER = "server";
    TFetcher::TDataWithInfo TFetcher::TranslateResp(const NPersQueue::TReadResponse& resp,
                                                    const TString& traceid) {
        TDataWithInfo res{
            .Data = TData{
                .Traceid = traceid,
            },
            .Cookie = resp.GetData().GetCookie(),
        };

        res.Data.Messages.reserve(resp.GetData().MessageBatchSize());
        for (const NPersQueue::ReadResponse::Data::MessageBatch& batch : resp.GetData().GetMessageBatch()) {
            TTopicData data;
            data.Topic = batch.GetTopic();

            data.Data.reserve(batch.MessageSize());
            res.Messages += batch.MessageSize();
            for (const NPersQueue::ReadResponse::Data::Message& m : batch.GetMessage()) {
                TString d = m.GetData();

                TChunk::TMeta meta{
                    .CreateTime = TInstant::MilliSeconds(m.GetMeta().GetCreateTimeMs()),
                };

                for (const NPersQueue::KeyValue& kv : m.GetMeta().GetExtraFields().items()) {
                    if (kv.key() == EXTRA_FIELD_FILE) {
                        meta.File = kv.value();
                    }

                    if (kv.key() == EXTRA_FIELD_SERVER) {
                        meta.Server = kv.value();
                    }
                }

                res.DataSize += d.size();
                data.Data.emplace_back(NLb::TChunk{
                    .Meta = std::move(meta),
                    .Data = std::move(d),
                });
            }
            res.Data.Messages.push_back(std::move(data));
        }

        return res;
    }

    void TFetcher::Cleanup() {
        auto sharedId = std::make_shared<TString>(Settings_->UnistatSignalId);
        while (!Messages_.empty()) {
            try {
                Messages_.front().Future.Subscribe([uncommittedMemory = Settings_->UncommittedMemory,
                                                    id = sharedId,
                                                    size = Messages_.front().DataSize](const auto&) {
                    uncommittedMemory->Release(*id, size);
                });
                Messages_.pop_front();
            } catch (const std::exception& e) {
                TLog::Error("LbReader exception at fetcher cleanup. %s", e.what());
            }
        }
    }
}
