#include "reader_impl.h"

#include "fetcher.h"

#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/unistat/builder.h>

namespace NPassport::NLb {
    static NPersQueue::TPQLibSettings CreatePqLibSettings(const TReaderSettings& srcSettings) {
        NPersQueue::TPQLibSettings settings;
        settings.ThreadsCount = srcSettings.Threads;
        settings.CompressionPoolThreads = srcSettings.Threads;
        return settings;
    }

    TReader::TImpl::TImpl(const TReaderSettings& settings)
        : Settings_(settings)
        , Pq_(CreatePqLibSettings(Settings_))
        , Logger_(new TLogger(Settings_.LogLevel))
        , PqSettings_(TranslateSettings(Settings_))
        , Backoff_(TDuration::Seconds(1), TDuration::Seconds(30), 2, 0.01)
        , IsRuning_(false)
        , DataSize_("lb_" + settings.UnistatSignalId + "_bytes")
        , Errors_("lb_" + settings.UnistatSignalId + "_errors")
        , Timestat_("lb_" + settings.UnistatSignalId + "_commitdelay",
                    NUnistat::TTimeStat::CreateBoundsFromMaxValue(TDuration::Minutes(10)))
    {
        Settings_.UncommittedMemory->AddResource(
            Settings_.UnistatSignalId,
            Settings_.MaxUncommittedSize,
            Settings_.MaxUncommittedSize * Settings_.ReserveRatio);
    }

    void TReader::TImpl::StartLoop(TReader::TLoopFunc func) {
        try {
            IsRuning_ = true;

            std::unique_ptr<TFetcher> fetcher;

            while (IsRunning()) {
                try {
                    if (!fetcher) {
                        fetcher = MakeNewFetcher();
                    }

                    DataSize_ += fetcher->Process(func);
                    Backoff_.Decrease();
                    fetcher->WaitForAWhile();
                    continue;
                } catch (const std::exception& e) {
                    TLog::Warning("LbReader exception: %s", e.what());
                }

                fetcher.reset();
                ++Errors_;
                TLog::Warning() << "LbReader (" << GetServer() << ") got error."
                                << " Sleep for " << Backoff_.GetCurrentValue();
                Backoff_.Sleep();
                Backoff_.Increase();
            }

            if (fetcher) {
                fetcher->ForceCommit();
            }
        } catch (const std::exception& e) {
            TLog::Error("LbReader exception. toplvl: %s", e.what());
        }
        IsRuning_ = false;
    }

    void TReader::TImpl::StopLoop() {
        Backoff_.Interrupt();
        IsRuning_ = false;
    }

    void TReader::TImpl::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(DataSize_);
        builder.Add(Errors_);
        Timestat_.AddUnistat(builder);
    }

    bool TReader::TImpl::IsOk() const {
        return IsRunning() &&
               TInstant::Now() - PrevFetcherCreation_.load(std::memory_order_relaxed) > Settings_.FailThreshold;
    }

    const TString& TReader::TImpl::GetServer() const {
        return Settings_.ServerName;
    }

    bool TReader::TImpl::IsRunning() const {
        return IsRuning_.load(std::memory_order_relaxed);
    }

    std::unique_ptr<TFetcher> TReader::TImpl::MakeNewFetcher() {
        PrevFetcherCreation_.store(LastFetcherCreation_, std::memory_order_relaxed);
        LastFetcherCreation_ = TInstant::Now();

        return std::make_unique<TFetcher>(Pq_, PqSettings_, Settings_, Logger_, Timestat_);
    }

    NPersQueue::TConsumerSettings TReader::TImpl::TranslateSettings(const TReaderSettings& sets) {
        NPersQueue::TConsumerSettings res;
        res.Topics = sets.Topics;
        res.ReadMirroredPartitions = sets.ReadMirroredPartitions;
        res.Server.Address = sets.ServerName;
        if (sets.ServerPort) {
            res.Server.Port = *sets.ServerPort;
        }
        res.ClientId = sets.ClientId;
        res.MaxCount = sets.ReadMaxCounts;
        res.MaxSize = sets.ReadMaxBytes;
        res.MaxInflyRequests = sets.InflightReads;
        res.MaxMemoryUsage = sets.MaxMemoryUsage;

        if (sets.TvmClient) { // empty in tests
            res.CredentialsProvider = NPersQueue::CreateTVMCredentialsProvider(
                sets.TvmClient,
                Logger_,
                sets.TvmDstAlias);
        }

        return res;
    }

}
