#include "resource_dispatcher.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/yexception.h>

namespace NPassport::NLb {
    TResourceDispatcher::TResource::TResource(const TString& unistatPrefix, ui64 limit, ui64 reserve)
        : Limit(NUtils::CreateStr(unistatPrefix, "_limit"), NUnistat::NSuffix::AVVV)
        , Used(NUtils::CreateStr(unistatPrefix, "_used"), NUnistat::NSuffix::AVVV)
        , Reserve(reserve)
    {
        Limit = limit;
    }

    ui64 TResourceDispatcher::TResource::GetRequiredReserve(ui64 required) const {
        return required + Reserve;
    }

    TResourceDispatcher::TResourceDispatcher(const TString& unistatPrefix)
        : UnistatPrefix_(unistatPrefix)
        , TotalLimit_(NUtils::CreateStr(unistatPrefix, "_limit"), NUnistat::NSuffix::AVVV)
        , TotalUsed_(NUtils::CreateStr(unistatPrefix, "_used"), NUnistat::NSuffix::AVVV)
        , TotalRequired_(NUtils::CreateStr(unistatPrefix, "_required"), NUnistat::NSuffix::AVVV)
    {
    }

    void TResourceDispatcher::AddResource(const TString& id, ui64 limit, ui64 reserve) {
        auto newResource = std::make_unique<TResource>(
            NUtils::CreateStr(UnistatPrefix_, '_', id),
            limit,
            reserve);

        auto [it, ok] = Resources_.emplace(id, std::move(newResource));
        Y_ENSURE(ok, "Resource dispatcher id '" << id << "' already exist");

        TotalLimit_ += limit;
        RequestReserve(*it->second, 0);
    }

    void TResourceDispatcher::AddUnistat(NUnistat::TBuilder& builder) {
        for (const auto& [_, resource] : Resources_) {
            builder.Add(resource->Limit);
            builder.Add(resource->Used);
        }
        builder.Add(TotalLimit_);
        builder.Add(TotalUsed_);
        builder.Add(TotalRequired_);
    }

    /**
     * (1) - limit_ is the local limit, available for the resource.
     * (2) - requested_ is the counter, used to restrict excessive borrowing by other resources and
     * to request the release of what has already been borrowed:
     * other resources can only borrow from (limit_ - requested_)
     * requested_ is updated on every call of tryAcquire or release, so that resource always has a reserve.
     * (3) - used_ is the local counter of the successfully acquired resource.
     *
     * used_ < limit_:                           used_ > limit_:
     *      ______________________________                         __________
     *     |______________________________|            |__________|__________|
     *     | may borrow | reserved | used 0            | borrowed |   used   0
     *    (1)          (2)        (3)                 (3)        (1)
     *                                                           (2)
     *
     * When calling tryAcquire within the resource limit (used_ + size <= limit_),
     * only check that there is enough unused resource (not borrowed by others).
     *
     * When tryAcquire goes over limit (used_ + size > limit_),
     * we can try to borrow from others ('may borrow' area) if it not already fully borrowed.
     */

    static ui64 GetBorrowedOf(ui64 size, ui64 used, ui64 limit) {
        if (used <= limit) {
            return 0;
        }

        return std::min(size, used - limit);
    }

    bool TResourceDispatcher::TryAcquire(const TString& id, ui64 size) {
        TResource& resource = GetResource(id);

        if (size == 0) {
            return true;
        }

        std::unique_lock lock(Mutex_);

        ui64 required = resource.Used.GetValue() + size;
        RequestReserve(resource, required);

        ui64 borrow = GetBorrowedOf(size, required, resource.Limit.GetValue());
        if (borrow == 0) {
            if (!TryUse(size)) {
                return false;
            }
        } else {
            if (!TryBorrow(size, borrow)) {
                if (resource.Used.GetValue() != 0) {
                    return false;
                }

                // Request for size > resource.limit_, borrow attempt failed
                // Request must be somehow satisfied, even if it leads to exceeding the limits,
                // so that resource consumer doesn't get stuck on it.
                // Request is satisfied if the resource local limit is available:
                if (!TryForceBorrow(size, borrow)) {
                    return false;
                }

                TLog::Warning() << "Resource dispatcher acquire " << size << " for " << id
                                << " which is more than the limit (" << resource.Limit.GetValue() << ")";
            }
        }

        resource.Used += size;

        return true;
    }

    bool TResourceDispatcher::Release(const TString& id, ui64 size) {
        TResource& resource = GetResource(id);

        std::unique_lock lock(Mutex_);

        if (size > resource.Used.GetValue()) {
            TLog::Warning() << "Resource dispatcher trying to release more then requested for " << id
                            << ": releasing " << size << " when used " << resource.Used.GetValue()
                            << " of " << resource.Limit.GetValue();
            return false;
        }

        Release(size, GetBorrowedOf(size, resource.Used.GetValue(), resource.Limit.GetValue()));
        ReleaseReserve(resource, resource.Used.GetValue());

        resource.Used -= size;

        return true;
    }

    TResourceDispatcher::TResource& TResourceDispatcher::GetResource(const TString& id) {
        auto it = Resources_.find(id);
        Y_ENSURE(it != Resources_.end(), "Resource dispatcher unknown id: " << id);

        return *it->second;
    }

    void TResourceDispatcher::RequestReserve(TResource& resource, ui64 required) {
        ui64 newRequested = std::min(resource.GetRequiredReserve(required),
                                     resource.Limit.GetValue());
        if (newRequested <= resource.Requested) {
            return;
        }

        TotalRequired_ += newRequested - resource.Requested;
        resource.Requested = newRequested;
    }

    void TResourceDispatcher::ReleaseReserve(TResource& resource, ui64 required) {
        ui64 newRequested = resource.GetRequiredReserve(required);
        if (newRequested >= resource.Requested) {
            return;
        }

        TotalRequired_ -= resource.Requested - newRequested;
        resource.Requested = newRequested;
    }

    bool TResourceDispatcher::TryUse(ui64 size) {
        if (TotalUsed_.GetValue() + size > TotalLimit_.GetValue()) {
            return false;
        }

        TotalUsed_ += size;

        return true;
    }

    bool TResourceDispatcher::TryBorrow(ui64 size, ui64 borrow) {
        if (TotalUsed_.GetValue() + size > TotalLimit_.GetValue() ||
            TotalRequired_.GetValue() + borrow > TotalLimit_.GetValue()) {
            return false;
        }

        TotalRequired_ += borrow;
        TotalUsed_ += size;

        return true;
    }

    bool TResourceDispatcher::TryForceBorrow(ui64 size, ui64 borrow) {
        if (TotalUsed_.GetValue() + size - borrow > TotalLimit_.GetValue()) {
            return false;
        }

        TotalRequired_ += borrow;
        TotalUsed_ += size;

        return true;
    }

    void TResourceDispatcher::Release(ui64 used, ui64 borrowed) {
        TotalUsed_ -= used;
        TotalRequired_ -= borrowed;
    }
}
