#include <passport/infra/libs/cpp/logbroker/resource_dispatcher/resource_dispatcher.h>

#include <library/cpp/testing/unittest/registar.h>

#include <vector>

using namespace NPassport;
using namespace NPassport::NLb;

Y_UNIT_TEST_SUITE(ResourceDispatcher) {
    struct TResourceSettings {
        TString Id;
        ui64 Limit = 0;
        ui64 Reserve = 0;
    };

    static std::unique_ptr<TResourceDispatcher> MakeDispatcher(
        const std::vector<TResourceSettings>& settings) {
        std::unique_ptr<TResourceDispatcher> dispatcher = std::make_unique<TResourceDispatcher>("test");
        for (const auto& resourceSettings : settings) {
            dispatcher->AddResource(resourceSettings.Id,
                                    resourceSettings.Limit,
                                    resourceSettings.Reserve);
        }

        return dispatcher;
    }

    Y_UNIT_TEST(addResource) {
        const TString testId = "testId";
        const TString fakeId = "fakeId";

        TResourceDispatcher dispatcher("test");
        dispatcher.AddResource(testId, 100);

        UNIT_ASSERT_EXCEPTION_CONTAINS(dispatcher.AddResource(testId, 100),
                                       std::exception,
                                       "Resource dispatcher id '" + testId + "' already exist");

        UNIT_ASSERT_NO_EXCEPTION(dispatcher.TryAcquire(testId, 0));
        UNIT_ASSERT_NO_EXCEPTION(dispatcher.Release(testId, 0));

        UNIT_ASSERT_EXCEPTION_CONTAINS(dispatcher.TryAcquire(fakeId, 0),
                                       std::exception,
                                       "Resource dispatcher unknown id");
        UNIT_ASSERT_EXCEPTION_CONTAINS(dispatcher.Release(fakeId, 0),
                                       std::exception,
                                       "Resource dispatcher unknown id");
    }

    Y_UNIT_TEST(staticGuaranteeTryAcquire) {
        const TString testUid = "testUid";

        std::vector<TResourceSettings> settings;
        settings.push_back(TResourceSettings{
            .Id = testUid,
            .Limit = 1000,
        });

        std::unique_ptr<TResourceDispatcher> dispatcher;

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 1000));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 400));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 400));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 201));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 200));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 0));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 500));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 0));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 500));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 0));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));

        // Запрос размера сверх установленного лимита должен срабатывать,
        // если на момент запроса ничего не используется:
        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 2000));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));
    }

    Y_UNIT_TEST(staticGuaranteeRelease) {
        const TString testUid = "testUid";

        std::vector<TResourceSettings> settings;
        settings.push_back(TResourceSettings{
            .Id = testUid,
            .Limit = 1000,
        });

        std::unique_ptr<TResourceDispatcher> dispatcher;

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 1000));
        UNIT_ASSERT(dispatcher->Release(testUid, 1000));

        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 1000));
        UNIT_ASSERT(dispatcher->Release(testUid, 500));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 501));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 500));

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(!dispatcher->Release(testUid, 300));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 600));
        UNIT_ASSERT(dispatcher->Release(testUid, 300));
        UNIT_ASSERT(!dispatcher->Release(testUid, 301));
        UNIT_ASSERT(dispatcher->Release(testUid, 300));
        UNIT_ASSERT(!dispatcher->Release(testUid, 1));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 1000));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->Release(testUid, 0));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 1000));
        UNIT_ASSERT(dispatcher->Release(testUid, 0));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));
        UNIT_ASSERT(dispatcher->Release(testUid, 1000));

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 2000));
        UNIT_ASSERT(dispatcher->Release(testUid, 2000));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 1000));

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 2000));
        UNIT_ASSERT(dispatcher->Release(testUid, 1000));
        UNIT_ASSERT(!dispatcher->TryAcquire(testUid, 1));
        UNIT_ASSERT(dispatcher->Release(testUid, 500));
        UNIT_ASSERT(dispatcher->TryAcquire(testUid, 500));
    }

    Y_UNIT_TEST(staticGuaranteeForMultipleResources) {
        const TString firstId = "firstId";
        const TString secondId = "secondId";
        const TString thirdId = "thirdId";

        TResourceDispatcher dispatcher("test");
        dispatcher.AddResource(firstId, 500);
        dispatcher.AddResource(secondId, 1000);
        dispatcher.AddResource(thirdId, 5000);

        UNIT_ASSERT(dispatcher.TryAcquire(firstId, 500));
        UNIT_ASSERT(dispatcher.TryAcquire(secondId, 1000));
        UNIT_ASSERT(dispatcher.TryAcquire(thirdId, 5000));

        UNIT_ASSERT(!dispatcher.TryAcquire(firstId, 1));
        UNIT_ASSERT(!dispatcher.TryAcquire(secondId, 1));
        UNIT_ASSERT(!dispatcher.TryAcquire(thirdId, 1));

        UNIT_ASSERT(!dispatcher.Release(firstId, 501));
        UNIT_ASSERT(!dispatcher.Release(secondId, 1001));
        UNIT_ASSERT(!dispatcher.Release(thirdId, 5001));

        UNIT_ASSERT(dispatcher.Release(firstId, 500));
        UNIT_ASSERT(dispatcher.Release(secondId, 1000));
        UNIT_ASSERT(dispatcher.Release(thirdId, 5000));
    }

    static ui64 AcquireAll(TResourceDispatcher& dispatcher, const TString& id, ui64 batchSize) {
        ui64 acquiredTotal = 0;
        while (dispatcher.TryAcquire(id, batchSize)) {
            acquiredTotal += batchSize;
        }

        return acquiredTotal;
    }

    Y_UNIT_TEST(overlimit) {
        const TString borrower = "borrower";
        const TString lender = "lender";

        std::vector<TResourceSettings> settings;
        settings.push_back(TResourceSettings{
            .Id = borrower,
            .Limit = 500,
        });
        settings.push_back(TResourceSettings{
            .Id = lender,
            .Limit = 500,
            .Reserve = 100,
        });

        std::unique_ptr<TResourceDispatcher> dispatcher;

        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->Release(borrower, AcquireAll(*dispatcher, borrower, 40)));
        UNIT_ASSERT(dispatcher->TryAcquire(borrower, 500));
        UNIT_ASSERT(dispatcher->TryAcquire(lender, 500));

        // Даже одалживая, нельзя переваливать за общий лимит:
        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT_GE(1000, AcquireAll(*dispatcher, borrower, 40) + AcquireAll(*dispatcher, lender, 40));

        // borrower не должен захватывать зарезервированный объем у lender:
        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT_GE(900, AcquireAll(*dispatcher, borrower, 40));
        UNIT_ASSERT(dispatcher->TryAcquire(lender, 100));
        UNIT_ASSERT(!dispatcher->TryAcquire(lender, 500));

        dispatcher = MakeDispatcher(settings);
        dispatcher->TryAcquire(lender, 200);
        UNIT_ASSERT_GE(700, AcquireAll(*dispatcher, borrower, 40));
        UNIT_ASSERT(dispatcher->TryAcquire(lender, 100));
        UNIT_ASSERT(!dispatcher->TryAcquire(lender, 300));

        // borrower не должен захватывать объем, на который претендует lender:
        dispatcher = MakeDispatcher(settings);
        ui64 borrowerAcquired = AcquireAll(*dispatcher, borrower, 40);
        dispatcher->TryAcquire(lender, 500);
        dispatcher->Release(borrower, borrowerAcquired);
        UNIT_ASSERT_GE(500, AcquireAll(*dispatcher, borrower, 40));

        // Запрос размера свех установленного лимита должен срабатывать, если доступен весь личный лимит,
        // не зависимо от того, можно ли на него одолжить:
        dispatcher = MakeDispatcher(settings);
        UNIT_ASSERT(dispatcher->TryAcquire(borrower, 1200));

        dispatcher = MakeDispatcher(settings);
        dispatcher->TryAcquire(lender, 300);
        UNIT_ASSERT(dispatcher->TryAcquire(borrower, 1200));

        dispatcher = MakeDispatcher(settings);
        dispatcher->TryAcquire(borrower, 300);
        UNIT_ASSERT(!dispatcher->TryAcquire(borrower, 1200));

        dispatcher = MakeDispatcher(settings);
        dispatcher->TryAcquire(borrower, 600);
        UNIT_ASSERT(!dispatcher->TryAcquire(lender, 1200));
    }
}
