#include <passport/infra/libs/cpp/logbroker/resource_dispatcher/resource_dispatcher.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

#include <atomic>
#include <queue>
#include <thread>
#include <unordered_map>
#include <vector>

using namespace NPassport;
using namespace NPassport::NLb;

Y_UNIT_TEST_SUITE(ResourceDispatcher) {
    struct TRequest {
        ui64 Size = 0;
        TDuration ArrivalDelay;    // Время задержки до прихода запроса.
        TDuration ProcessingDelay; // Время на обработку запроса.

        ui64 Repeated = 1; // Число повторений запроса (для простоты составления сценариев)
    };

    struct TResourceScript {
        ui64 Limit = 0;
        ui64 Reserve = 0;
        std::vector<TRequest> Requests;
    };

    struct TCase {
        TDuration Timeout = TDuration::Seconds(1);
        ui64 RequiredLoopsNumber = 10;
        std::unordered_map<TString, TResourceScript> Scripts;
    };

    class TTestRunner {
    public:
        void Run(const TCase& expCase) {
            auto dispatcher = std::make_shared<NLb::TResourceDispatcher>("experiment");
            for (const auto& script : expCase.Scripts) {
                dispatcher->AddResource(script.first, script.second.Limit, script.second.Reserve);
            }

            StartTime_ = TInstant::Now();

            Error_ = false;
            Running_ = true;
            ReadyCounter_ = 0;

            std::vector<std::thread> threads;
            for (const auto& script : expCase.Scripts) {
                threads.emplace_back(std::thread([&]() {
                    RunRequests(dispatcher, script.first, script.second.Requests, expCase.RequiredLoopsNumber);
                }));
            }

            while (ReadyCounter_ != expCase.Scripts.size()) {
                if (Error_ || TInstant::Now() >= StartTime_ + expCase.Timeout) {
                    break;
                }

                Sleep(expCase.Timeout / 1000);
            }

            Running_ = false;
            for (auto& thr : threads) {
                thr.join();
            }

            UNIT_ASSERT(!Error_.load());
        }

    private:
        void RunRequests(std::shared_ptr<NLb::TResourceDispatcher> dispatcher,
                         const TString& id,
                         const std::vector<TRequest>& requests,
                         ui64 requiredLoopsNumber) {
            TInstant nextRequestTime = StartTime_ + requests[0].ArrivalDelay;

            using TReleaseTimeSizePair = std::pair<TInstant, ui64>;
            std::priority_queue<TReleaseTimeSizePair, std::vector<TReleaseTimeSizePair>, std::greater<>> queueOut;

            ui64 loopsNumber = 0;
            size_t requestIdx = 0;
            ui64 requestRepeated = 0;
            while (Running_) {
                TInstant now = TInstant::Now();

                while (!queueOut.empty() && queueOut.top().first <= now) {
                    if (!dispatcher->Release(id, queueOut.top().second)) {
                        SendError(id, "release failed");
                        return;
                    }

                    queueOut.pop();
                }

                while (nextRequestTime <= now &&
                       dispatcher->TryAcquire(id, requests[requestIdx].Size)) {
                    queueOut.emplace(now + requests[requestIdx].ProcessingDelay,
                                     requests[requestIdx].Size);

                    if (++requestRepeated >= requests[requestIdx].Repeated) {
                        requestRepeated = 0;
                        if (++requestIdx == requests.size()) {
                            requestIdx = 0;
                            if (++loopsNumber == requiredLoopsNumber) {
                                ++ReadyCounter_;
                            }
                        }
                    }

                    nextRequestTime = now + requests[requestIdx].ArrivalDelay;
                }

                Sleep(std::min(nextRequestTime, queueOut.empty() ? nextRequestTime : queueOut.top().first) - now);
            }

            while (!queueOut.empty()) {
                if (!dispatcher->Release(id, queueOut.top().second)) {
                    SendError(id, "final release failed");
                    return;
                }

                queueOut.pop();
            }

            if (loopsNumber < requiredLoopsNumber) {
                SendError(id, "not finished required number of loops");
                return;
            }
        }

        void SendError(const TString& id, const TString& err) {
            Cerr << NUtils::CreateStr("=== error occurred in resource '", id, "' : ", err, '\n');
            Error_ = true;
            Running_ = false;
        }

    private:
        TInstant StartTime_;

        std::atomic_bool Error_;
        std::atomic_bool Running_;
        std::atomic<ui64> ReadyCounter_;
    };

    static const TResourceScript& GetLenderScript() {
        static const TResourceScript script{
            .Limit = 1200,
            .Reserve = 100,
            .Requests = {
                TRequest{
                    .Size = 100,
                    .ArrivalDelay = TDuration::MicroSeconds(100),
                    .ProcessingDelay = TDuration::MicroSeconds(600),
                    .Repeated = 10},
                TRequest{
                    .Size = 400,
                    .ArrivalDelay = TDuration::MicroSeconds(400),
                    .ProcessingDelay = TDuration::MicroSeconds(100)},
                TRequest{
                    .Size = 200,
                    .ArrivalDelay = TDuration::MicroSeconds(200),
                    .ProcessingDelay = TDuration::MicroSeconds(400),
                    .Repeated = 5},
                TRequest{
                    .Size = 100,
                    .ArrivalDelay = TDuration::MicroSeconds(400),
                    .ProcessingDelay = TDuration::MicroSeconds(100)}}};

        return script;
    }

    static const TResourceScript& GetBorrowerScript() {
        static const TResourceScript script{
            .Limit = 600,
            .Reserve = 600,
            .Requests = {
                TRequest{
                    .Size = 200,
                    .ArrivalDelay = TDuration::MicroSeconds(0),
                    .ProcessingDelay = TDuration::MicroSeconds(800),
                    .Repeated = 10},
                TRequest{
                    .Size = 600,
                    .ArrivalDelay = TDuration::MicroSeconds(400),
                    .ProcessingDelay = TDuration::MicroSeconds(100)},
                TRequest{
                    .Size = 400,
                    .ArrivalDelay = TDuration::MicroSeconds(0),
                    .ProcessingDelay = TDuration::MicroSeconds(400),
                    .Repeated = 5},
                TRequest{
                    .Size = 600,
                    .ArrivalDelay = TDuration::MicroSeconds(400),
                    .ProcessingDelay = TDuration::MicroSeconds(100)}}};

        return script;
    }

    Y_UNIT_TEST(threeLenders) {
        TTestRunner{}.Run(TCase{
            .RequiredLoopsNumber = 10,
            .Scripts = {
                {"lender_1", GetLenderScript()},
                {"lender_2", GetLenderScript()},
                {"lender_3", GetLenderScript()}}});
    }

    Y_UNIT_TEST(twoLendersOneBorrower) {
        TTestRunner{}.Run(TCase{
            .RequiredLoopsNumber = 10,
            .Scripts = {
                {"lender_1", GetLenderScript()},
                {"lender_2", GetLenderScript()},
                {"borrower", GetBorrowerScript()}}});
    }

    Y_UNIT_TEST(oneLenderTwoBorrowers) {
        TTestRunner{}.Run(TCase{
            .RequiredLoopsNumber = 10,
            .Scripts = {
                {"lender", GetLenderScript()},
                {"borrower_1", GetBorrowerScript()},
                {"borrower_2", GetBorrowerScript()}}});
    }

    Y_UNIT_TEST(threeBorrowers) {
        TTestRunner{}.Run(TCase{
            .RequiredLoopsNumber = 10,
            .Scripts = {
                {"borrower_1", GetBorrowerScript()},
                {"borrower_2", GetBorrowerScript()},
                {"borrower_3", GetBorrowerScript()}}});
    }
}
