#include "csv_row.h"

#include <util/stream/str.h>

namespace NPassport::NRowParsers {
    TCsvRow::TCsvRow(TStringBuf line) {
        SplitToFields(line, ' ', '`', Fields_);
    }

    TStringBuf TCsvRow::GetField(size_t idx) const {
        if (idx >= Fields_.size()) {
            return {};
        }

        return Fields_[idx].GetBuf();
    }

    const TString& TCsvRow::GetAllocatedField(size_t idx) const {
        Y_ENSURE(idx < Fields_.size(), "idx=" << idx << ". size=" << Fields_.size());
        return Fields_[idx].GetAllocated();
    }

    TString TCsvRow::PrintDebug() const {
        TStringStream res;

        res << Endl;
        for (const TField& f : Fields_) {
            res << f.GetBuf() << Endl;
        }

        return res.Str();
    }

    TField TCsvRow::Unquote(TStringBuf& line, char quote) {
        TString res;

        Y_ENSURE(line.SkipPrefix(TStringBuf(&quote, 1)), "field must starts with quote char");
        auto lastAdded = line.begin();

        for (auto it = line.begin(); it != line.end(); ++it) {
            if (*it == quote) {
                auto next = it;
                ++next;

                if (next != line.end() && *next == quote) {
                    if (res.empty()) {
                        res.reserve(line.size());
                    }
                    res.append(lastAdded, next);
                    lastAdded = next + 1;
                    ++it;
                } else {
                    if (res) {
                        res.append(lastAdded, it);
                        line.Skip(next - line.begin());
                        return TField(std::move(res));
                    }

                    TStringBuf b(line.begin(), it);
                    line.Skip(next - line.begin());
                    return TField(b);
                }
            }
        }

        ythrow yexception() << "field must ends with quote char";
    }

    void TCsvRow::SplitToFields(TStringBuf line, char delimeter, char quote, TFields& fields) {
        while (line) {
            if (line.StartsWith(quote)) {
                fields.push_back(TField(Unquote(line, quote)));
                Y_ENSURE(line.empty() || line.SkipPrefix(TStringBuf(&delimeter, 1)),
                         "quoted field must be at the end or be before delimeter");
            } else {
                fields.push_back(TField(line.NextTok(delimeter)));
            }
        }
    }
}
