#pragma once

#include "tskv_parser.h"

#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>

#include <array>
#include <optional>

namespace NPassport::NRowParsers {
    namespace NPrivate {
        enum class ENoKeys {
            Count = 0,
        };
    }

    template <typename EKey = NPrivate::ENoKeys>
    class TFieldsHolder {
    public:
        using TTokens = THashMap<TStringBuf, EKey>;
        using TExtraFields = THashMap<TStringBuf, TStringBuf>;

        explicit TFieldsHolder(const TTokens& tokens = {}, bool saveExtraFields = true)
            : Tokens_(tokens)
            , SaveExtraFields_(saveExtraFields)
        {
        }

        void ResetFromTskv(TStringBuf line, const TTskvPolicy& policy = {}) {
            TTskvParser::Parse(line, *this, Buf_, policy);
        }

        TStringBuf Get(EKey key) const {
            TStringBuf res = GetOptional((size_t)key);
            Y_ENSURE(res, "empty field '" << key << "'");
            return res;
        }

        TStringBuf GetDefault(EKey key, TStringBuf def = {}) const {
            TStringBuf res = GetOptional((size_t)key);
            return res ? res : def;
        }

        TStringBuf GetOptional(EKey key) const {
            return GetOptional((size_t)key);
        }

        bool Has(EKey key) const {
            return (bool)GetOptional((size_t)key);
        }

        TStringBuf Get(TStringBuf key) const {
            auto res = GetOptional(key);
            Y_ENSURE(res, "no field '" << key << "'");
            return *res;
        }

        TStringBuf GetDefault(TStringBuf key, TStringBuf def = {}) const {
            auto res = GetOptional(key);
            return res.has_value() ? *res : def;
        }

        std::optional<TStringBuf> GetOptional(TStringBuf key) const {
            if (auto it = ExtraFields_.find(key); it != ExtraFields_.end()) {
                return it->second;
            }

            if (auto it = Tokens_.find(key); it != Tokens_.end()) {
                return GetOptional(it->second);
            }

            return {};
        }

        bool Has(TStringBuf key) const {
            return GetOptional(key).has_value();
        }

        const TExtraFields& ExtraFields() const {
            return ExtraFields_;
        }

        void clear() {
            Fields_.fill(TStringBuf());
            ExtraFields_.clear();
        }

        TStringBuf& operator[](EKey key) {
            return operator[]((size_t)key);
        }

        TStringBuf& operator[](size_t key) {
            Y_ENSURE(key < (size_t)EKey::Count);
            return Fields_[key];
        }

        TStringBuf& operator[](TStringBuf key) {
            if (auto it = Tokens_.find(key); it != Tokens_.end()) {
                return operator[](it->second);
            }

            return SaveExtraFields_ ? ExtraFields_[key] : Trash_;
        }

    private:
        TStringBuf GetOptional(size_t key) const {
            Y_ENSURE(key < (size_t)EKey::Count);
            return Fields_[key];
        }

    private:
        const TTokens& Tokens_;

        std::array<TStringBuf, (size_t)EKey::Count> Fields_;

        bool SaveExtraFields_ = true;
        TExtraFields ExtraFields_;
        TStringBuf Trash_;

        TString Buf_;
    };
}
