#include "tskv_row.h"

#include "utils.h"

#include <passport/infra/libs/cpp/utils/ipaddr.h>

#include <util/generic/yexception.h>
#include <util/stream/str.h>

namespace NPassport::NRowParsers {
    TTskvRow::TTskvRow(TStringBuf line, const TTskvPolicy& policy) {
        Reset(line, policy);
    }

    TTskvRow::TTskvRow(const TTskvRow::TFields& fields)
        : Fields_(fields)
    {
    }

    void TTskvRow::Reset(TStringBuf line, const TTskvPolicy& policy) {
        TTskvParser::Parse(line, Fields_, Buf_, policy);
    }

    TStringBuf TTskvRow::GetField(TStringBuf name) const {
        TStringBuf res = GetOptionalField(name);
        Y_ENSURE(res, "no field '" << name << "'");
        return res;
    }

    TStringBuf TTskvRow::GetOptionalField(TStringBuf key) const {
        auto it = Fields_.find(key);
        return it == Fields_.end() ? TStringBuf() : it->second;
    }

    NUtils::TIpAddr TTskvRow::GetIp(TStringBuf key) const {
        TStringBuf value = GetField(key);
        NUtils::TIpAddr res;
        Y_ENSURE(res.Parse(TString(value)),
                 "invalid ip address in '" << key << "': '" << value << "'");
        return res;
    }

    std::optional<NUtils::TIpAddr> TTskvRow::GetOptionalIp(TStringBuf key) const {
        TStringBuf value = GetOptionalField(key);
        if (value.empty()) {
            return {};
        }

        NUtils::TIpAddr res;
        Y_ENSURE(res.Parse(TString(value)),
                 "invalid ip address in '" << key << "': '" << value << "'");
        return res;
    }

    TInstant TTskvRow::GetUnixtime(TStringBuf key) const {
        double value;
        GetNum(key, value);
        return TUtils::TimestampFromFloat(value);
    }

    std::optional<TInstant> TTskvRow::GetOptionalUnixtime(TStringBuf key) const {
        double value;
        if (!GetOptionalNum(key, value)) {
            return {};
        }
        return TUtils::TimestampFromFloat(value);
    }

    TString TTskvRow::PrintDebug() const {
        TStringStream res;

        res << Endl;
        for (const auto& [key, value] : Fields_) {
            res << key << " : " << value << Endl;
        }

        return res.Str();
    }
}
