#include <passport/infra/libs/cpp/logbroker/row_parsers/csv_row.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NRowParsers;

Y_UNIT_TEST_SUITE(RowParserCsv) {
    Y_UNIT_TEST(RowCtx) {
        UNIT_ASSERT_VALUES_EQUAL(
            TCsvRow("1 2011-12-08T11:19:27.434273+04 7F passport 11806301 - - web ses_kill ssl=1;aid=1323266326487:1600958337:127;ttl=0 95.108.173.242 - 1699279861298040776 http://mda-dev.yandex.ru/login?retpath=https%3A%2F%2Fpassportdev-7.yandex.ru%2Fpassport%3Fmode%3Dpassport&ncrnd=584709 - `Mozilla/5.0 (Ubuntu; X11; Linux x86_64; rv:8.0) Gecko/20100101 Firefox/8.0`")
                .PrintDebug(),
            R"(
1
2011-12-08T11:19:27.434273+04
7F
passport
11806301
-
-
web
ses_kill
ssl=1;aid=1323266326487:1600958337:127;ttl=0
95.108.173.242
-
1699279861298040776
http://mda-dev.yandex.ru/login?retpath=https%3A%2F%2Fpassportdev-7.yandex.ru%2Fpassport%3Fmode%3Dpassport&ncrnd=584709
-
Mozilla/5.0 (Ubuntu; X11; Linux x86_64; rv:8.0) Gecko/20100101 Firefox/8.0
)");

        UNIT_ASSERT_VALUES_EQUAL(
            TCsvRow("1 2011-12-08T11:19:27.434273+04 7F passport 11806301 - - web ses_kill ssl=1;aid=1323266326487:1600958337:127;ttl=0 95.108.173.242 - 1699279861298040776 `some test kek` - -")
                .PrintDebug(),
            R"(
1
2011-12-08T11:19:27.434273+04
7F
passport
11806301
-
-
web
ses_kill
ssl=1;aid=1323266326487:1600958337:127;ttl=0
95.108.173.242
-
1699279861298040776
some test kek
-
-
)");
    }

    Y_UNIT_TEST(unquote) {
        TStringBuf line;

        line = "`askjdadhjks`";
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "askjdadhjks");
        UNIT_ASSERT_VALUES_EQUAL(line, "");

        line = "`askjdadhjks` kasdfjabdfjks";
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "askjdadhjks");
        UNIT_ASSERT_VALUES_EQUAL(line, " kasdfjabdfjks");

        line = "`askjdadhjks` `kasdfjabdfjks`";
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "askjdadhjks");
        UNIT_ASSERT_VALUES_EQUAL(line, " `kasdfjabdfjks`");
        line.SkipPrefix(" ");
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "kasdfjabdfjks");
        UNIT_ASSERT_VALUES_EQUAL(line, "");

        line = "`askjd`adhjks`";
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "askjd");
        UNIT_ASSERT_VALUES_EQUAL(line, "adhjks`");

        line = "`askjd``adhjks`";
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "askjd`adhjks");
        UNIT_ASSERT_VALUES_EQUAL(line, "");

        line = "`ask``jdad``hjks`";
        UNIT_ASSERT_VALUES_EQUAL(TCsvRow::Unquote(line).GetBuf(), "ask`jdad`hjks");
        UNIT_ASSERT_VALUES_EQUAL(line, "");

        // Errors
        line = "askjdadhjks";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TCsvRow::Unquote(line),
                                       yexception,
                                       "field must starts with quote char");

        line = "`askjdadhjks";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TCsvRow::Unquote(line),
                                       yexception,
                                       "field must ends with quote char");

        line = "`askjdadhjks`asdsa";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TCsvRow(line),
                                       yexception,
                                       "quoted field must be at the end or be before delimeter");

        line = "`askjdadhjks` asdsa";
        UNIT_ASSERT_NO_EXCEPTION(TCsvRow(line));
    }

    Y_UNIT_TEST(field) {
        TString str = "somestr";

        TField f(std::move(str));
        UNIT_ASSERT_VALUES_EQUAL(f.GetBuf(), "somestr");
        UNIT_ASSERT_VALUES_EQUAL(f.GetAllocated(), "somestr");

        TStringBuf buf = "somestr2";
        f = TField(buf);
        UNIT_ASSERT_VALUES_EQUAL(f.GetBuf(), "somestr2");
        UNIT_ASSERT_VALUES_EQUAL((ui64)f.GetBuf().data(), (ui64)buf.data());
        UNIT_ASSERT_VALUES_EQUAL(f.GetAllocated(), "somestr2");
        UNIT_ASSERT_VALUES_UNEQUAL((ui64)f.GetAllocated().data(), (ui64)buf.data());
    }
}
