#include "fields_holder_ut.h"

#include <passport/infra/libs/cpp/logbroker/row_parsers/fields_holder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NRowParsers;

Y_UNIT_TEST_SUITE(RowParsersFieldsHolder) {
    using TTestHolder = TFieldsHolder<ETestKey>;

    static const TTestHolder::TTokens TEST_TOKENS = {
        {"foo", ETestKey::Foo},
        {"bar", ETestKey::Bar},
    };

    Y_UNIT_TEST(general) {
        TTestHolder holder(TEST_TOKENS);

        UNIT_ASSERT_EXCEPTION(holder.Get(ETestKey::Count), yexception);
        UNIT_ASSERT_EXCEPTION(holder[ETestKey::Count], yexception);

        UNIT_ASSERT(!holder.Has(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL("", holder.GetOptional(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL("some_default_value", holder.GetDefault(ETestKey::Foo, "some_default_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(holder.Get(ETestKey::Foo), yexception, "empty field 'Foo'");

        UNIT_ASSERT(holder.Has("foo"));
        UNIT_ASSERT(holder.GetOptional("foo"));
        UNIT_ASSERT_VALUES_EQUAL("", holder.GetOptional("foo").value());
        UNIT_ASSERT_VALUES_EQUAL("", holder.GetDefault("foo", "some_default_value"));
        UNIT_ASSERT_VALUES_EQUAL("", holder.Get("foo"));

        holder[ETestKey::Foo] = "foo_value";
        UNIT_ASSERT(holder.Has(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.GetOptional(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.GetDefault(ETestKey::Foo, "some_default_value"));
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.Get(ETestKey::Foo));

        UNIT_ASSERT(holder.Has("foo"));
        UNIT_ASSERT(holder.GetOptional("foo"));
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.GetOptional("foo").value());
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.GetDefault("foo", "some_default_value"));
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.Get("foo"));

        UNIT_ASSERT_VALUES_EQUAL(0, holder.ExtraFields().size());
        UNIT_ASSERT(!holder.Has("some_field"));
        UNIT_ASSERT(!holder.GetOptional("some_field"));
        UNIT_ASSERT_VALUES_EQUAL("some_default_value", holder.GetDefault("some_field", "some_default_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(holder.Get("some_field"), yexception, "no field 'some_field'");

        holder["some_field"] = "some_value";
        UNIT_ASSERT_VALUES_EQUAL(1, holder.ExtraFields().size());
        UNIT_ASSERT(holder.ExtraFields().contains("some_field"));
        UNIT_ASSERT_VALUES_EQUAL("some_value", holder.ExtraFields().find("some_field")->second);

        UNIT_ASSERT(holder.Has("some_field"));
        UNIT_ASSERT(holder.GetOptional("some_field"));
        UNIT_ASSERT_VALUES_EQUAL("some_value", holder.GetOptional("some_field").value());
        UNIT_ASSERT_VALUES_EQUAL("some_value", holder.GetDefault("some_field", "some_default_value"));
        UNIT_ASSERT_VALUES_EQUAL("some_value", holder.Get("some_field"));
    }

    Y_UNIT_TEST(doNotSaveExtraFields) {
        TTestHolder holder(TEST_TOKENS, false);

        holder[ETestKey::Foo] = "foo_value";
        holder["some_field"] = "some_value";

        UNIT_ASSERT(holder.Has(ETestKey::Foo));
        UNIT_ASSERT(!holder.Has("some_field"));
    }

    Y_UNIT_TEST(parseFromTskv) {
        TTestHolder holder(TEST_TOKENS);

        holder.ResetFromTskv("tskv\tfoo=foo_value\tsome_field=some_value");

        UNIT_ASSERT(holder.Has(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.Get(ETestKey::Foo));

        UNIT_ASSERT(!holder.Has(ETestKey::Bar));

        UNIT_ASSERT_VALUES_EQUAL(1, holder.ExtraFields().size());
        UNIT_ASSERT(holder.ExtraFields().contains("some_field"));
        UNIT_ASSERT_VALUES_EQUAL("some_value", holder.ExtraFields().find("some_field")->second);

        UNIT_ASSERT(holder.Has("some_field"));
        UNIT_ASSERT_VALUES_EQUAL("some_value", holder.Get("some_field"));

        holder.ResetFromTskv("tskv\tfoo=");
        UNIT_ASSERT(!holder.Has(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL(0, holder.ExtraFields().size());
        UNIT_ASSERT(!holder.Has("some_field"));

        holder.ResetFromTskv("tskv\tfoo=foo_value\tbar=bar_value\tsome_field=with\\tunescape");
        UNIT_ASSERT_VALUES_EQUAL("foo_value", holder.Get(ETestKey::Foo));
        UNIT_ASSERT_VALUES_EQUAL("bar_value", holder.Get(ETestKey::Bar));
        UNIT_ASSERT_VALUES_EQUAL("with\tunescape", holder.Get("some_field"));
    }

    Y_UNIT_TEST(parseFromTskvWithoutExtraFields) {
        TTestHolder holder(TEST_TOKENS, false);

        holder.ResetFromTskv("tskv\tbar=bar_value\tsome_field=some_value");

        UNIT_ASSERT(holder.Has(ETestKey::Bar));
        UNIT_ASSERT_VALUES_EQUAL("bar_value", holder.Get(ETestKey::Bar));

        UNIT_ASSERT(!holder.Has(ETestKey::Foo));

        UNIT_ASSERT_VALUES_EQUAL(0, holder.ExtraFields().size());
        UNIT_ASSERT(!holder.Has("some_field"));
    }
}
