#include <passport/infra/libs/cpp/logbroker/row_parsers/tskv_row.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NRowParsers;

Y_UNIT_TEST_SUITE(RowParserTskv) {
    Y_UNIT_TEST(kvRow) {
        UNIT_ASSERT_VALUES_EQUAL(TTskvRow().PrintDebug(), "\n");
        UNIT_ASSERT_VALUES_EQUAL(TTskvRow().FieldsCount(), 0);

        // errors
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTskvRow(""),
                                       yexception,
                                       "line should start with 'tskv'");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TTskvRow("tskv\t"),
                                       yexception,
                                       "empty tskv line");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TTskvRow("tskv\t\tk=v\t"),
                                       yexception,
                                       "Incorrect tskv format");

        UNIT_ASSERT_NO_EXCEPTION(TTskvRow("tskv\tkey="));
        UNIT_ASSERT_NO_EXCEPTION(TTskvRow("tskv\t=val"));

        TTskvRow row("tskv\tuid=100500\tip=127.0.0.1\ttarget=message\toperation=delete\tunixtime=123456\tfoo=asd\\tfgh");
        UNIT_ASSERT_VALUES_EQUAL(
            row.PrintDebug(),
            R"(
foo : asd	fgh
ip : 127.0.0.1
operation : delete
target : message
uid : 100500
unixtime : 123456
)");

        UNIT_ASSERT_VALUES_EQUAL(row.FieldsCount(), 6);
        UNIT_ASSERT(row.HasField("uid"));
        UNIT_ASSERT_VALUES_EQUAL("100500", row.GetField("uid"));
        UNIT_ASSERT(row.HasField("ip"));
        UNIT_ASSERT_VALUES_EQUAL("127.0.0.1", row.GetField("ip"));
        UNIT_ASSERT(!row.HasField("date"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(row.GetField("date"),
                                       yexception,
                                       "no field 'date'");

        row.Reset("tskv\tabc=12345\tdef=1111\tabc=67890");
        UNIT_ASSERT_VALUES_EQUAL(row.FieldsCount(), 2);
        UNIT_ASSERT_VALUES_EQUAL("67890", row.GetField("abc"));
        UNIT_ASSERT_VALUES_EQUAL("1111", row.GetField("def"));
    }

    Y_UNIT_TEST(dontEscape) {
        TTskvRow row("tskv\tuid=100500\tip=127.0.0.1\ttarget=message\toperation=delete\tunixtime=123456\tfoo=asd\\tfgh",
                     TTskvPolicy{.Escape = ETskvEscapePolicy::KeepAsIs});
        UNIT_ASSERT_VALUES_EQUAL(
            row.PrintDebug(),
            R"(
foo : asd\tfgh
ip : 127.0.0.1
operation : delete
target : message
uid : 100500
unixtime : 123456
)");
    }

    Y_UNIT_TEST(checkPrefix) {
        TTskvRow row;

        for (ETskvPrefixPolicy p : {
                 ETskvPrefixPolicy::Required,
                 ETskvPrefixPolicy::SkipIfFound,
             }) {
            UNIT_ASSERT_NO_EXCEPTION(
                row.Reset("tskv\tuid=100500\tip=127.0.0.1",
                          TTskvPolicy{.Prefix = p}));
            UNIT_ASSERT_VALUES_EQUAL(
                row.PrintDebug(),
                R"(
ip : 127.0.0.1
uid : 100500
)");
        }

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            row.Reset("uid=100500\tip=127.0.0.1",
                      TTskvPolicy{.Prefix = ETskvPrefixPolicy::Required}),
            yexception,
            "line should start with 'tskv'");

        UNIT_ASSERT_NO_EXCEPTION(
            row.Reset("uid=100500\tip=127.0.0.1",
                      TTskvPolicy{.Prefix = ETskvPrefixPolicy::SkipIfFound}));
        UNIT_ASSERT_VALUES_EQUAL(
            row.PrintDebug(),
            R"(
ip : 127.0.0.1
uid : 100500
)");
    }
}
