#include "regex.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NRe2 {
    TRegex::TRegex(const TStringBuf r)
        : Regex_(re2::StringPiece(r.data(), r.size()))
    {
        Y_ENSURE(Regex_.ok(),
                 ""
                     << "invalid regex: '" << r << "'"
                     << ". error: " << Regex_.error()
                     << " at '" << Regex_.error_arg()
                     << "': error_code=" << (int)Regex_.error_code());
    }

    bool TRegex::FullMatch(TStringBuf in) const {
        return RE2::FullMatch(re2::StringPiece(in.data(), in.size()), Regex_);
    }

    bool TRegex::PartialMatch(TStringBuf in) const {
        return RE2::PartialMatch(re2::StringPiece(in.data(), in.size()), Regex_);
    }

    size_t TRegexGroups::NumberOfGroups() const {
        return Regex_.NumberOfCapturingGroups();
    }

    TPattern TPattern::WithClassicPlaceholders(TStringBuf pattern, size_t groupNumber) {
        TPlaceholders ph;
        ph.reserve(groupNumber);
        for (size_t idx = 0; idx < groupNumber; ++idx) {
            ph.push_back(NUtils::CreateStr("$", idx + 1));
        }

        return WithCustomPlaceholders(pattern, ph);
    }

    TPattern TPattern::WithCustomPlaceholders(TStringBuf pattern, const TPlaceholders& ph) {
        TPattern res;
        res.Parts_ = ParseParts(pattern, ph);
        res.GroupNumber_ = ph.size();
        return res;
    }

    TPattern::TParts TPattern::ParseParts(TStringBuf str, const TPlaceholders& ph) {
        if (!str) {
            return {};
        }

        TParts res = {{TString(str), 0}};
        for (size_t idx = ph.size(); idx > 0; --idx) {
            TParts tmp = std::move(res);
            res.clear(); // for clang-tidy

            for (const auto& pair : tmp) {
                TStringBuf s(pair.first);

                if (s.empty()) {
                    res.push_back(pair);
                    continue;
                }

                while (s) {
                    TStringBuf left;
                    TStringBuf right;

                    if (s.TrySplit(ph[idx - 1], left, right)) {
                        res.push_back({TString(left), idx});
                    } else {
                        break;
                    }

                    s = right;
                }

                res.push_back({TString(s), pair.second});
            }
        }

        if (std::pair<TString, ui32>{"", 0} == res.back()) {
            res.pop_back();
        }

        return res;
    }

    TReplacingRegex::TReplacingRegex(TStringBuf in, TStringBuf out)
        : Regex_(in)
        , Pattern_(NRe2::TPattern::WithClassicPlaceholders(out, Regex_.NumberOfGroups()))
    {
        Y_ENSURE(!Pattern_.IsEmpty(), "pattern cannot be empty");
    }

    bool TReplacingRegex::Apply(TStringBuf field, TString& out) const {
        TRegexGroupsCtx ctx;
        TSmallVec<TStringBuf> groups;
        return Apply(ctx, groups, field, out);
    }
}
