#pragma once

#include <contrib/libs/re2/re2/re2.h>

#include <library/cpp/containers/stack_vector/stack_vec.h>

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>

#include <vector>

namespace NPassport::NRe2 {
    class TRegex {
    public:
        TRegex(const TStringBuf r);

        bool FullMatch(TStringBuf in) const;
        bool PartialMatch(TStringBuf in) const;

    protected:
        const re2::RE2 Regex_;
    };

    class TRegexGroupsCtx {
        using TFunc = bool (*)(const re2::StringPiece&, const RE2&, const re2::RE2::Arg* const[], int);

    public:
        template <class Cont>
        bool Match(const re2::RE2& regex, TStringBuf in, Cont& out, TFunc func) {
            out.clear();
            Bufs_.clear();
            Args_.clear();
            ArgsPtr_.clear();

            Bufs_.resize(regex.NumberOfCapturingGroups());
            for (int idx = 0; idx < regex.NumberOfCapturingGroups(); ++idx) {
                Args_.push_back(&Bufs_[idx]);
                ArgsPtr_.push_back(&Args_.back());
            }

            if (!func(re2::StringPiece(in.data(), in.size()),
                      regex,
                      ArgsPtr_.data(),
                      ArgsPtr_.size()))
            {
                return false;
            }

            out.reserve(Bufs_.size());
            for (re2::StringPiece sp : Bufs_) {
                out.push_back(typename Cont::value_type(sp.data(), sp.size()));
            }

            return true;
        }

    private:
        TSmallVec<re2::StringPiece> Bufs_;
        TSmallVec<re2::RE2::Arg> Args_;
        TSmallVec<re2::RE2::Arg*> ArgsPtr_;
    };

    class TRegexGroups: public TRegex {
    public:
        using TRegex::FullMatch;
        using TRegex::PartialMatch;
        using TRegex::TRegex;

        template <class Cont>
        bool FullMatch(TRegexGroupsCtx& ctx, TStringBuf in, Cont& out) const {
            return ctx.Match<Cont>(Regex_, in, out, &RE2::FullMatchN);
        }

        template <class T = TStringBuf>
        bool FullMatch(TStringBuf in, std::vector<T>& out) const {
            TRegexGroupsCtx ctx;
            return FullMatch(ctx, in, out);
        }

        template <class Cont>
        bool PartialMatch(TRegexGroupsCtx& ctx, TStringBuf in, Cont& out) const {
            return ctx.Match<Cont>(Regex_, in, out, &RE2::PartialMatchN);
        }

        template <class T = TStringBuf>
        bool PartialMatch(TStringBuf in, std::vector<T>& out) const {
            TRegexGroupsCtx ctx;
            return PartialMatch(ctx, in, out);
        }

        size_t NumberOfGroups() const;
    };

    class TPattern {
    public:
        // $1, $2, $3, ...
        static TPattern WithClassicPlaceholders(TStringBuf pattern, size_t groupNumber);

        // {abc}, $qwe, $asd$, ...
        using TPlaceholders = std::vector<TString>;
        static TPattern WithCustomPlaceholders(TStringBuf pattern, const TPlaceholders& ph);

        bool IsEmpty() const {
            return Parts_.empty();
        }

        template <class Cont>
        TString BuildString(const Cont& vals) const {
            TString res;
            BuildString(vals, res);
            return res;
        }

        template <class Cont>
        void BuildString(const Cont& vals, TString& res) const {
            Y_ENSURE(vals.size() == GroupNumber_,
                     "expected " << GroupNumber_ << " vals, got " << vals.size());
            res.clear();

            for (const auto& [substring, pos] : Parts_) {
                res.append(substring);
                if (pos != 0) {
                    res.append(vals.at(pos - 1));
                }
            }
        }

    protected:
        TPattern() = default;

        using TParts = std::vector<std::pair<TString, ui32>>;

        static TParts ParseParts(TStringBuf str, const TPlaceholders& ph);
        TParts Parts_;
        size_t GroupNumber_ = 0;
    };

    class TReplacingRegex {
    public:
        TReplacingRegex(TStringBuf in, TStringBuf out);

        template <class Cont>
        bool Apply(TRegexGroupsCtx& ctx, Cont& groups, TStringBuf field, TString& out) const {
            if (!Regex_.PartialMatch(ctx, field, groups)) {
                return false;
            }

            Pattern_.BuildString(groups, out);
            return true;
        }

        bool Apply(TStringBuf field, TString& out) const;

    private:
        NRe2::TRegexGroups Regex_;
        NRe2::TPattern Pattern_;
    };
}
