#include <passport/infra/libs/cpp/re2/regex.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NRe2;

Y_UNIT_TEST_SUITE(Regex) {
    Y_UNIT_TEST(ctor) {
        UNIT_ASSERT_NO_EXCEPTION(TRegex(""));

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRegex("("),
                                       yexception,
                                       "invalid regex: '('. error: missing ): ( at '('");
    }

    Y_UNIT_TEST(simplePartialMatch) {
        TRegex r(R"(\b2\d\d\b)");
        UNIT_ASSERT(r.PartialMatch("  239  "));
        UNIT_ASSERT(!r.PartialMatch("  339  "));

        UNIT_ASSERT(!r.FullMatch("  239  "));
        UNIT_ASSERT(!r.FullMatch("  339  "));
    }

    Y_UNIT_TEST(simpleFullMatch) {
        TRegex r(R"(  \b2\d\d\b  )");
        UNIT_ASSERT(r.PartialMatch("  239  "));
        UNIT_ASSERT(!r.PartialMatch("  339  "));

        UNIT_ASSERT(r.FullMatch("  239  "));
        UNIT_ASSERT(!r.FullMatch("  339  "));
    }

    Y_UNIT_TEST(partialGroups) {
        TRegexGroups r(R"(\b(2)\d(\d)\b)");
        UNIT_ASSERT(!r.FullMatch("  239  "));
        UNIT_ASSERT(!r.FullMatch("  339  "));

        UNIT_ASSERT(!r.PartialMatch("  339  "));

        std::vector<TString> vec;
        UNIT_ASSERT(r.PartialMatch("  239  ", vec));
        UNIT_ASSERT_VALUES_EQUAL(2, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("2", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("9", vec[1]);
    }

    Y_UNIT_TEST(fullGroups) {
        TRegexGroups r(R"(  \b(2)\d(\d)\b  )");
        std::vector<TString> vec;
        UNIT_ASSERT(r.FullMatch("  239  ", vec));
        UNIT_ASSERT_VALUES_EQUAL(2, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("2", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("9", vec[1]);
        vec.clear();

        UNIT_ASSERT(!r.FullMatch("  339  "));
        UNIT_ASSERT(!r.PartialMatch("  339  "));

        UNIT_ASSERT(r.PartialMatch("  239  ", vec));
        UNIT_ASSERT_VALUES_EQUAL(2, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("2", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("9", vec[1]);
    }

    Y_UNIT_TEST(parsePattern) {
        class TPatternTest: public TPattern {
        public:
            using TPattern::ParseParts;
            using TPattern::TParts;
        };

        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.", 2}, {".mega_", 1}}),
                                 TPatternTest::ParseParts(R"(some.$2.mega_$1)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.", 2}, {".mega_", 1}, {" signal", 0}}),
                                 TPatternTest::ParseParts(R"(some.$2.mega_$1 signal)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.", 2}, {".me", 2}, {"ga_", 1}, {" signal", 0}}),
                                 TPatternTest::ParseParts(R"(some.$2.me$2ga_$1 signal)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"", 2}, {".me", 2}, {"ga_", 1}}),
                                 TPatternTest::ParseParts(R"($2.me$2ga_$1)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"", 1}, {".me", 2}, {"ga_", 1}}),
                                 TPatternTest::ParseParts(R"($1.me$2ga_$1)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"", 2}, {".me", 1}, {"ga_", 2}}),
                                 TPatternTest::ParseParts(R"($2.me$1ga_$2)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.", 2}, {".mega_", 1}, {" sig", 2}, {"nal", 0}}),
                                 TPatternTest::ParseParts(R"(some.$2.mega_$1 sig$2nal)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.", 2}, {"", 1}, {" sig", 2}, {"nal", 0}}),
                                 TPatternTest::ParseParts(R"(some.$2$1 sig$2nal)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.", 1}, {"", 2}, {" sig", 2}, {"nal", 0}}),
                                 TPatternTest::ParseParts(R"(some.$1$2 sig$2nal)", {"$1", "$2"}));

        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.$.mega_", 1}}),
                                 TPatternTest::ParseParts(R"(some.$.mega_$1)", {"$1", "$2"}));
        UNIT_ASSERT_VALUES_EQUAL(TPatternTest::TParts({{"some.$2.me$2ga_", 1}, {" signal", 0}}),
                                 TPatternTest::ParseParts(R"(some.$2.me$2ga_$1 signal)", {"$1"}));
    }

    Y_UNIT_TEST(patternClassic) {
        TPattern pat = TPattern::WithClassicPlaceholders("some.$2.mega_$1 sig$2nal", 2);

        UNIT_ASSERT_EXCEPTION_CONTAINS(pat.BuildString(std::vector<TString>({})),
                                       yexception,
                                       "expected 2 vals, got 0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pat.BuildString(std::vector<TStringBuf>({"", "kke", "bar"})),
                                       yexception,
                                       "expected 2 vals, got 3");

        UNIT_ASSERT_VALUES_EQUAL("some.bar.mega_kke sigbarnal", pat.BuildString(std::vector<TString>({"kke", "bar"})));
    }

    Y_UNIT_TEST(patternCustom) {
        TPattern pat = TPattern::WithCustomPlaceholders("some.{abc}.mega_$asd$ sig$qwenal", {"{abc}", "$qwe", "$asd$"});

        UNIT_ASSERT_EXCEPTION_CONTAINS(pat.BuildString(std::vector<TString>({})),
                                       yexception,
                                       "expected 3 vals, got 0");
        UNIT_ASSERT_EXCEPTION_CONTAINS(pat.BuildString(std::vector<TString>({"", "kke", "bar", "lol"})),
                                       yexception,
                                       "expected 3 vals, got 4");

        UNIT_ASSERT_VALUES_EQUAL("some.kke.mega_foo sigbarnal", pat.BuildString(std::vector<TString>({"kke", "bar", "foo"})));
    }

    Y_UNIT_TEST(replacingRegex) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TReplacingRegex("", ""),
                                       yexception,
                                       "pattern cannot be empty");

        TReplacingRegex r(R"(http(s)?://([^/]+)/)", "some.$2.mega_$1 sig$2nal");

        TString str;
        UNIT_ASSERT(!r.Apply("lskjfslkjfbkldsa", str));
        UNIT_ASSERT_VALUES_EQUAL("", str);

        UNIT_ASSERT(r.Apply("https://yandex.ru/", str));
        UNIT_ASSERT_VALUES_EQUAL("some.yandex.ru.mega_s sigyandex.runal", str);
    }
}
