#pragma once

#include <util/datetime/base.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/string/builder.h>
#include <util/thread/pool.h>

#include <map>
#include <mutex>

namespace NPassport::NReporter {
    class TMapper {
    public:
        template <class Parser, class Stats>
        static std::map<TString, Stats> Run(const TVector<TString>& filenames, size_t threads);

        enum class EPolicy {
            Exclude,
            Include,
        };

        static TVector<TString> GetLogList(const TString& storage,
                                           const TString& date,
                                           const TString& filename,
                                           const std::vector<TString>& exclude = {},
                                           EPolicy policy = EPolicy::Exclude);
    };

    template <class Parser, class Stats>
    std::map<TString, Stats> TMapper::Run(const TVector<TString>& filenames, size_t threads) {
        TThreadPool queue;
        queue.Start(threads);

        std::map<TString, Stats> res;
        std::mutex m;

        for (const TString& file : filenames) {
            queue.SafeAddFunc([&m, &res, file]() {
                try {
                    Cout << (TStringBuilder() << "File started: " << file << Endl);
                    TInstant start = TInstant::Now();

                    Parser parser(file);
                    parser.Parse();
                    Stats stats = parser.FetchStats();

                    TDuration dur = TInstant::Now() - start;
                    Cout << (TStringBuilder() << "File finished: " << file
                                              << ". Took: " << dur
                                              << ". Size: " << parser.GetSize() / 1024. / 1024 / 1024 << " Gb"
                                              << Endl);

                    std::unique_lock lock(m);
                    res.emplace(file, std::move(stats));
                } catch (const std::exception& e) {
                    Cerr << "Failed to parse file: "
                         << file << ". "
                         << e.what() << Endl;
                }
            });
        }

        Cout << "Waiting for parsing" << Endl;
        queue.Stop();
        Cout << "Parsing finished" << Endl;

        return res;
    }
}
