#pragma once

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>
#include <util/stream/str.h>

#include <optional>
#include <time.h>

namespace NPassport::NCommon {
    class TCookie {
        static constexpr time_t MAX_EXPIRES_TIME = std::numeric_limits<i32>::max();

    public:
        enum class ESameSite {
            Strict,
            Lax,
            None,
        };

        TCookie(const TString& name, const TString& value)
            : Secure_(false)
            , HttpOnly_(false)
            , Encode_(false)
            , Expires_(0)
            , Name_(name)
            , Value_(value)
            , Path_("/")
        {
        }

        TCookie(const TCookie& cookie) = default;
        TCookie& operator=(const TCookie& cookie) = default;

        bool operator<(const TCookie& cookie) const {
            return Name() < cookie.Name();
        }

        const TString& Name() const {
            return Name_;
        }

        const TString& Value() const {
            return Value_;
        }

        void SetSecure(bool value) {
            Secure_ = value;
        }

        bool Secure() const {
            return Secure_;
        }

        void SetExpires(time_t value) {
            Expires_ = value;
        }

        time_t Expires() const {
            return Expires_;
        }

        void SetPermanent(bool value) {
            Expires_ = value ? MAX_EXPIRES_TIME : 0;
        }

        bool Permanent() const {
            return Expires_ == MAX_EXPIRES_TIME;
        }

        void SetHttpOnly(bool value) {
            HttpOnly_ = value;
        }

        bool HttpOnly() const {
            return HttpOnly_;
        }

        const TString& Path() const {
            return Path_;
        }

        void SetPath(const TString& value) {
            Path_ = value;
        }

        const TString& Domain() const {
            return Domain_;
        }

        void SetDomain(const TString& value) {
            Domain_ = value;
        }

        std::optional<ESameSite> SameSite() const {
            return SameSite_;
        }

        void SetSameSite(std::optional<ESameSite> s) {
            SameSite_ = s;
        }

        TString ToString() const {
            TString res;

            TStringOutput stream(res);
            stream << Name_;
            stream << '=' << (Encode_ ? NUtils::Urlencode(Value_) : Value_);
            if (!Domain_.empty()) {
                stream << "; domain=" << Domain_;
            }
            if (!Path_.empty()) {
                stream << "; path=" << Path_;
            }
            if (Expires_) {
                stream << "; expires=" << FormatTime(Expires_);
            }
            if (SameSite_) {
                stream << "; SameSite=" << *SameSite_;
            }
            if (Secure_) {
                stream << "; secure";
            }
            if (HttpOnly_) {
                stream << "; HttpOnly";
            }
            return res;
        }

        bool UrlEncode() const {
            return Encode_;
        }

        void SetUrlEncode(bool value) {
            Encode_ = value;
        }

    protected:
        static TString FormatTime(time_t value) {
            return TInstant::Seconds(value).ToRfc822String();
        }

    private:
        bool Secure_;
        bool HttpOnly_;
        bool Encode_;
        time_t Expires_;
        TString Name_, Value_, Path_, Domain_;
        std::optional<ESameSite> SameSite_;
    };

}
