#include "request.h"

#include <passport/infra/libs/cpp/request/cookie.h>

#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/cookies/cookies.h>
#include <library/cpp/http/server/response.h>

#include <util/network/address.h>

#include <set>

namespace NPassport::NHttp {
    static const TString COOKIES_ = "Cookie";
    static const TString EMPTY_;
    static const TString HOST_ = "Host";
    static const TString HTTPS_ = "X-HTTPS-Req";
    static const TString SET_COOKIE_ = "Set-Cookie";
    static const TString REMOTE_ADDR_ = "X-Real-IP";
    static const TString REQUEST_ID_ = "X-Request-Id";
    static const TString AUTHORIZATION_ = "Authorization";
    static const NCommon::TExtendedArg EMPTY_ARG_;

    static const std::set<TString> ALLOWED_METHODS_ = {
        "CONNECT",
        "DELETE",
        "GET",
        "HEAD",
        "OPTIONS",
        "PATCH",
        "POST",
        "PUT",
        "TRACE"};

    TRequest::TRequest(const TRequestReplier::TReplyParams& params, const TRequestReplier& parent)
        : Params_(params)
        , Parent_(parent)
        , FirstLine_(params.Input.FirstLine())
        , RequestIdGuard_(nullptr)
    {
        InitHeaders();
        InitRequestId();

        ScanCgi(FirstLine_.Cgi, EArgSource::Get);

        Method_ = FirstLine_.Method;
        if (ALLOWED_METHODS_.find(Method_) == ALLOWED_METHODS_.end()) {
            throw TMethodException(Method_);
        }

        if (Method_ == "POST") {
            Body_ = params.Input.ReadAll();
        } else if (Method_ == "GET") {
            params.Input.ReadAll(); // to clean up socket
        }
    }

    TRequest::~TRequest() = default;

    static const TString HEADER_CONTENT_LENGTH = "Content-Length";
    static const TString ZERO_ = "0";
    void TRequest::Flush(const TStackVec<THttpInputHeader, 2>& extraHeaders) {
        THttpResponse resp(Out_.Code);
        resp.SetContent(Out_.Body);
        if (Out_.Body.empty()) {
            resp.AddHeader(THttpInputHeader(HEADER_CONTENT_LENGTH, ZERO_));
        }

        for (auto& h : Out_.Headers) {
            resp.AddHeader(h);
        }

        for (auto& h : extraHeaders) {
            resp.AddHeader(h);
        }

        for (const auto& [name, value] : Out_.Cookies) {
            resp.AddHeader(THttpInputHeader(SET_COOKIE_, value.ToString()));
        }

        resp.OutTo(Params_.Output);
    }

    bool TRequest::HasHeader(const TString& name) const {
        return Headers_.find(name) != Headers_.end();
    }

    const TString& TRequest::GetHeader(const TString& name) const {
        auto it = Headers_.find(name);
        return it == Headers_.end() ? EMPTY_ : it->second;
    }

    NCommon::THttpHeaders TRequest::GetAllHeaders() const {
        return Headers_;
    }

    bool TRequest::HasCookie(const TString& name) const {
        LazyCookies();
        return Lazy_.Cookies.find(name) != Lazy_.Cookies.end();
    }

    const TString& TRequest::GetCookie(const TString& name) const {
        LazyCookies();
        auto it = Lazy_.Cookies.find(name);
        return it == Lazy_.Cookies.end() ? EMPTY_ : it->second;
    }

    bool TRequest::HasArg(const TString& name) const {
        return Cgi_.find(name) != Cgi_.end();
    }

    const TString& TRequest::GetArg(const TString& name) const {
        auto it = Cgi_.find(name);
        return it == Cgi_.end() ? EMPTY_ : it->second.Value;
    }

    const NCommon::TExtendedArg& TRequest::GetExtendedArg(const TString& name) const {
        auto it = Cgi_.find(name);
        return it == Cgi_.end() ? EMPTY_ARG_ : it->second;
    }

    void TRequest::ArgNames(std::vector<TString>& v) const {
        v.clear();
        v.reserve(Cgi_.size());
        for (const auto& [name, value] : Cgi_) {
            v.push_back(name);
        }
        std::sort(v.begin(), v.end());
        v.erase(std::unique(v.begin(), v.end()), v.end());
    }

    const TString& TRequest::GetRemoteAddr() const {
        LazyRemoteAddr();
        return Lazy_.RemoteAddr;
    }

    TString TRequest::GetUri() const {
        TString res = GetPath();
        const TString& query_string = GetQueryString();
        if (!query_string.empty()) {
            res.push_back('?');
            res.append(query_string);
        }
        return res;
    }

    const TString& TRequest::GetPath() const {
        LazyPath();
        return Lazy_.Path;
    }

    const TString& TRequest::GetQueryString() const {
        LazyQueryString();
        return Lazy_.QueryString;
    }

    const TString& TRequest::GetRequestMethod() const {
        return Method_;
    }

    const TString& TRequest::GetRequestId() const {
        return ReqId_;
    }

    void TRequest::ScanCgiFromBody() {
        ScanCgi(Body_, EArgSource::Body);
    }

    void TRequest::ForceProvideRequestId() {
        if (ReqId_) {
            return;
        }

        ReqId_ = NUtils::Bin2hex(NUtils::TCrypto::RandBytes(8));
        NUtils::SetThreadLocalRequestId(&ReqId_);
    }

    TRequest::TDuplicatedArgs TRequest::GetDuplicatedArgs() const {
        if (Cgi_.empty()) {
            return {};
        }

        TDuplicatedArgs res;

        // https://en.cppreference.com/w/cpp/named_req/UnorderedAssociativeContainer
        // "std::unordered_multiset and std::unordered_multimap ...
        // can have multiple elements with the same key
        // (which must always be adjacent on iterations)"

        auto prev = Cgi_.begin();

        for (auto it = ++Cgi_.begin(); it != Cgi_.end();) {
            if (it->first != prev->first) {
                prev = it;
                ++it;
                continue;
            }

            res.reserve(Cgi_.size());
            res.emplace_back(prev->first, prev->second.Value);
            do {
                res.emplace_back(it->first, it->second.Value);
            } while (++it != Cgi_.end() && it->first == prev->first);
        }

        return res;
    }

    bool TRequest::IsSecure() const {
        // Proxed from nginx
        return GetHeader(HTTPS_) == "on";
    }

    const TString& TRequest::GetHost() const {
        return GetHeader(HOST_);
    }

    bool TRequest::IsBodyEmpty() const {
        return Body_.empty();
    }

    void TRequest::SetStatus(HttpCodes code) {
        Out_.Code = code;
    }

    void TRequest::SetHeader(const TString& name, const TString& value) {
        Out_.Headers.push_back(THttpInputHeader(name, value));
    }

    void TRequest::SetCookie(const NCommon::TCookie& cookie) {
        auto it = Out_.Cookies.find(cookie.Name());
        if (it == Out_.Cookies.end()) {
            Out_.Cookies.emplace(cookie.Name(), cookie);
        } else {
            it->second = cookie;
        }
    }

    void TRequest::Write(const TString& body) {
        if (Out_.Body.empty()) {
            Out_.Body = body;
        } else {
            Out_.Body.append(body);
        }
    }

    HttpCodes TRequest::GetStatusCode() const {
        return Out_.Code;
    }

    size_t TRequest::GetResponseSize() const {
        return Out_.Body.size();
    }

    const TString& TRequest::GetRequestBody() const {
        return Body_;
    }

    TStringBuf TRequest::GetRequestCgi() const {
        return FirstLine_.Cgi;
    }

    void TRequest::InitHeaders() {
        Headers_.reserve(Params_.Input.Headers().Count());
        for (const THttpInputHeader& h : Params_.Input.Headers()) {
            Headers_.emplace(h.Name(), h.Value());
        }
    }

    void TRequest::LazyCookies() const {
        if (!Lazy_.Cookies.empty()) {
            return;
        }

        auto it = Headers_.find(COOKIES_);
        if (it == Headers_.end()) {
            return;
        }

        const THttpCookies cookies(it->second);
        Lazy_.Cookies.reserve(cookies.Size());
        for (const auto& [name, value] : cookies) {
            Lazy_.Cookies.emplace(name, value);
        }
    }

    void TRequest::LazyPath() const {
        if (!Lazy_.Path.empty()) {
            return;
        }

        Lazy_.Path = FirstLine_.Path;
    }

    void TRequest::LazyQueryString() const {
        if (!Lazy_.QueryString.empty()) {
            return;
        }

        Lazy_.QueryString = FirstLine_.Cgi;
    }

    void TRequest::LazyRemoteAddr() const {
        if (!Lazy_.RemoteAddr.empty()) {
            return;
        }

        // Proxed from nginx
        Lazy_.RemoteAddr = GetHeader(REMOTE_ADDR_);

        if (Lazy_.RemoteAddr.empty()) {
            NAddr::TOpaqueAddr remoteAddr;
            if (getpeername(Parent_.Socket(), remoteAddr.MutableAddr(), remoteAddr.LenPtr()) != 0) {
                ythrow TSystemError() << "HasLocalAddress: getpeername() failed. ";
            }
            Lazy_.RemoteAddr = NAddr::PrintHost(remoteAddr);
        }
    }

    void TRequest::InitRequestId() {
        // Proxed from nginx
        const TString& h = GetHeader(REQUEST_ID_);
        if (h.empty()) {
            return;
        }

        ReqId_ = h.substr(0, 16);
        NUtils::SetThreadLocalRequestId(&ReqId_);
    }

    void TRequest::ScanCgi(TStringBuf str, EArgSource source) {
        TCgiParameters cg;
        cg.ScanAddAll(str);

        Cgi_.reserve(cg.size() + Cgi_.size());
        for (const auto& [name, value] : cg) {
            Cgi_.emplace(
                name,
                NCommon::TExtendedArg(value, source == EArgSource::Get));
        }
    }
}
