#include <passport/infra/libs/cpp/request/http/request.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/buffer.h>
#include <util/stream/buffer.h>

using namespace NPassport;
using namespace NPassport::NHttp;

Y_UNIT_TEST_SUITE(PassportRequestHttp) {
    Y_UNIT_TEST(ci_map) {
        NCommon::THttpHeaders c;
        c["test"] = "test";
        c["Test1"] = "Test1";
        UNIT_ASSERT_VALUES_EQUAL("test", c["Test"]);
        UNIT_ASSERT_VALUES_EQUAL("", c["Tent"]);
        UNIT_ASSERT_VALUES_EQUAL("Test1", c["test1"]);

        NCommon::TCiStrHash h;
        UNIT_ASSERT_VALUES_EQUAL(320, h("test"));
        UNIT_ASSERT_VALUES_EQUAL(320, h("teST"));
        UNIT_ASSERT_VALUES_EQUAL(353, h("test!"));

        NCommon::TCiStrEqual e;
        UNIT_ASSERT(e("test", "Test"));
    }

    class TDummyReplier: public TRequestReplier {
    public:
        bool DoReply(const TReplyParams&) override {
            return true;
        }

    private:
    };

    class TSimpleStringInput: public IInputStream {
    public:
        TSimpleStringInput(const TString& string)
            : Str_(string)
        {
        }

    protected:
        size_t DoRead(void* buf, size_t len) override {
            Y_ASSERT(len != 0);

            if (Str_.size() < len) {
                len = Str_.size();
            }

            if (len == 0) {
                return 0;
            }

            for (size_t i = 0; i < len; ++i) {
                *(static_cast<char*>(buf) + i) = Str_[i];
            }

            Str_.remove(0, len);
            return len;
        }

    private:
        TString Str_;
    };

    Y_UNIT_TEST(request) {
        TDummyReplier replier;
        TSimpleStringInput input("POST /blackbox/?method=login&login=root HTTP/1.1\n"
                                 "User-Agent: Wget/1.13.4 (linux-gnu)\n"
                                 "Connection: Keep-Alive\n"
                                 "Cookie: foo=bar; yandex_login=ab=cd\n"
                                 "Content-Type: application/x-www-form-urlencoded\n"
                                 "X-Request-Id: 8f705a071c5dff247e8afee2eaeb0fcb\n"
                                 "Authorization: qwerty\n"
                                 "Content-Length: 62\n\n"
                                 "login=vasya&password=qwerty&otherparam=yes%26login%3Dimperator");
        TBufferOutput output;
        THttpInput http_input(&input);
        THttpOutput http_output(&output);
        TRequestReplier::TReplyParams params = {nullptr, http_input, http_output};

        TRequest r(params, replier);

        UNIT_ASSERT(!r.HasHeader("Host"));
        UNIT_ASSERT(r.HasHeader("Connection"));
        UNIT_ASSERT_VALUES_EQUAL("Keep-Alive", r.GetHeader("Connection"));
        UNIT_ASSERT(r.HasHeader("Authorization"));
        UNIT_ASSERT_VALUES_EQUAL("qwerty", r.GetHeader("Authorization"));

        UNIT_ASSERT(!r.HasCookie("Session_id"));
        UNIT_ASSERT(r.HasCookie("foo"));
        UNIT_ASSERT_VALUES_EQUAL("bar", r.GetCookie("foo"));

        std::vector<TString> v;
        UNIT_ASSERT(!r.HasArg("userip"));
        UNIT_ASSERT(!r.HasArg("password"));
        UNIT_ASSERT_VALUES_EQUAL("", r.GetArg("password"));
        UNIT_ASSERT_VALUES_EQUAL("", r.GetExtendedArg("password").Value);
        UNIT_ASSERT_VALUES_EQUAL(false, r.GetExtendedArg("password").IsGet);

        r.ArgNames(v);
        UNIT_ASSERT_VALUES_EQUAL(2, v.size());
        UNIT_ASSERT_VALUES_EQUAL("login", v[0]);
        UNIT_ASSERT_VALUES_EQUAL("method", v[1]);

        r.ScanCgiFromBody();
        UNIT_ASSERT(!r.HasArg("userip"));
        UNIT_ASSERT(r.HasArg("password"));
        UNIT_ASSERT_VALUES_EQUAL("qwerty", r.GetArg("password"));
        UNIT_ASSERT_VALUES_EQUAL("qwerty", r.GetExtendedArg("password").Value);
        UNIT_ASSERT_VALUES_EQUAL(false, r.GetExtendedArg("password").IsGet);
        r.ArgNames(v);
        UNIT_ASSERT_VALUES_EQUAL(4, v.size());
        UNIT_ASSERT_VALUES_EQUAL("login", v[0]);
        UNIT_ASSERT_VALUES_EQUAL("method", v[1]);
        UNIT_ASSERT_VALUES_EQUAL("otherparam", v[2]);
        UNIT_ASSERT_VALUES_EQUAL("password", v[3]);

        // UNIT_ASSERT_VALUES_EQUAL("", r.getRemoteAddr());
        UNIT_ASSERT_VALUES_EQUAL("/blackbox/?method=login&login=root", r.GetUri());
        UNIT_ASSERT_VALUES_EQUAL("/blackbox/", r.GetPath());
        UNIT_ASSERT_VALUES_EQUAL("method=login&login=root", r.GetQueryString());
        UNIT_ASSERT_VALUES_EQUAL("POST", r.GetRequestMethod());
        UNIT_ASSERT_VALUES_EQUAL("8f705a071c5dff24", r.GetRequestId());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetHost());
        UNIT_ASSERT(!r.IsSecure());
        UNIT_ASSERT(!r.IsBodyEmpty());

        UNIT_ASSERT_VALUES_EQUAL("http:///blackbox/?method=login&login=root", r.GetUrl());
        UNIT_ASSERT_VALUES_EQUAL("-", r.GetConsumerFormattedName());
        r.SetConsumerFormattedName("test_consumer");
        UNIT_ASSERT_VALUES_EQUAL("test_consumer", r.GetConsumerFormattedName());

        r.SetStatus(HTTP_NOT_FOUND);
        r.SetHeader("X-User_id", "123");
        r.SetCookie(NPassport::NCommon::TCookie("mda", "0"));
        r.Write("Hello");

        UNIT_ASSERT_VALUES_EQUAL(HTTP_NOT_FOUND, r.GetStatusCode());
        UNIT_ASSERT_VALUES_EQUAL(5, r.GetResponseSize());
        UNIT_ASSERT_STRINGS_EQUAL("login=vasya&password=qwerty&otherparam=yes%26login%3Dimperator",
                                  r.GetRequestBody());
        UNIT_ASSERT_VALUES_EQUAL("method=login&login=root", r.GetRequestCgi());

        r.Flush();

        TString result;
        output.Buffer().AsString(result);
        UNIT_ASSERT_VALUES_EQUAL("HTTP/1.1 404 Not found\r\n"
                                 "X-User_id: 123\r\n"
                                 "Set-Cookie: mda=0; path=/\r\n"
                                 "Content-Length: 5\r\n"
                                 "Connection: Close\r\n\r\n"
                                 "Hello",
                                 result);
    }

    Y_UNIT_TEST(request2) {
        TDummyReplier replier;
        TSimpleStringInput input("GET /blackbox/?method=userinfo&uid=11111 HTTP/1.1\n"
                                 "Host: blackbox.yandex.net\n"
                                 "X-HTTPS-Req: on\n"
                                 "X-Real-IP: 8.8.8.8\n"
                                 "Connection: Keep-Alive\n");
        TBufferOutput output;
        THttpInput http_input(&input);
        THttpOutput http_output(&output);
        TRequestReplier::TReplyParams params = {nullptr, http_input, http_output};

        TRequest r(params, replier);

        UNIT_ASSERT(r.HasHeader("Host"));
        UNIT_ASSERT_VALUES_EQUAL("blackbox.yandex.net", r.GetHeader("Host"));
        UNIT_ASSERT(r.HasHeader("Connection"));
        UNIT_ASSERT_VALUES_EQUAL("Keep-Alive", r.GetHeader("Connection"));

        UNIT_ASSERT(!r.HasCookie("Session_id"));

        std::vector<TString> v;
        UNIT_ASSERT(!r.HasArg("userip"));
        UNIT_ASSERT(r.HasArg("method"));
        UNIT_ASSERT_VALUES_EQUAL("userinfo", r.GetArg("method"));
        UNIT_ASSERT_VALUES_EQUAL("userinfo", r.GetExtendedArg("method").Value);
        UNIT_ASSERT_VALUES_EQUAL(true, r.GetExtendedArg("method").IsGet);

        UNIT_ASSERT_VALUES_EQUAL("8.8.8.8", r.GetRemoteAddr());
        UNIT_ASSERT_VALUES_EQUAL("8.8.8.8", r.GetRemoteAddr());
        UNIT_ASSERT_VALUES_EQUAL("/blackbox/?method=userinfo&uid=11111", r.GetUri());
        UNIT_ASSERT_VALUES_EQUAL("/blackbox/", r.GetPath());
        UNIT_ASSERT_VALUES_EQUAL("method=userinfo&uid=11111", r.GetQueryString());
        UNIT_ASSERT_VALUES_EQUAL("GET", r.GetRequestMethod());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetRequestId());
        UNIT_ASSERT_VALUES_EQUAL("blackbox.yandex.net", r.GetHost());
        UNIT_ASSERT(r.IsSecure());
        UNIT_ASSERT(r.IsBodyEmpty());
        UNIT_ASSERT_VALUES_EQUAL("https://blackbox.yandex.net/blackbox/?method=userinfo&uid=11111", r.GetUrl());

        r.ForceProvideRequestId();
        UNIT_ASSERT_VALUES_UNEQUAL("", r.GetRequestId());
        const TString reqid = r.GetRequestId();
        r.ForceProvideRequestId();
        UNIT_ASSERT_VALUES_EQUAL(reqid, r.GetRequestId());

        // set cookie twice
        r.SetStatus(HTTP_OK);
        r.SetCookie(NPassport::NCommon::TCookie("mda", "0"));
        r.SetCookie(NPassport::NCommon::TCookie("mda", "1"));

        UNIT_ASSERT_VALUES_EQUAL(HTTP_OK, r.GetStatusCode());
        UNIT_ASSERT_VALUES_EQUAL(0, r.GetResponseSize());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetRequestBody());
        UNIT_ASSERT_VALUES_EQUAL("method=userinfo&uid=11111", r.GetRequestCgi());

        r.Flush();

        TString result;
        output.Buffer().AsString(result);
        UNIT_ASSERT_VALUES_EQUAL("HTTP/1.1 200 Ok\r\n"
                                 "Content-Length: 0\r\n"
                                 "Set-Cookie: mda=1; path=/\r\n"
                                 "Connection: Close\r\n\r\n",
                                 result);
        r.Flush();
    }

    Y_UNIT_TEST(bad_request_method) {
        TDummyReplier replier;
        TSimpleStringInput input("QUACK /blackbox/?method=login&login=root HTTP/1.1\n"
                                 "Content-Length: 27\n\n"
                                 "login=vasya&password=qwerty");
        TBufferOutput output;
        THttpInput http_input(&input);
        THttpOutput http_output(&output);
        TRequestReplier::TReplyParams params = {nullptr, http_input, http_output};

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRequest(params, replier), TRequest::TMethodException, "QUACK");
    }

    Y_UNIT_TEST(deplicatedArgs) {
        struct TCase {
            TString Params;
            TRequest::TDuplicatedArgs Dups;
        };

        std::vector<TCase> cases = {
            {
                .Params = "",
                .Dups = {},
            },
            {
                .Params = "arg1=val11&arg1=val12&arg1=val13",
                .Dups = {
                    {"arg1", "val11"},
                    {"arg1", "val12"},
                    {"arg1", "val13"},
                },
            },
            {
                .Params = "arg1=val11&arg1=val12&arg2=val21&arg1=val13&arg3=val31&arg2=val22",
                .Dups = {
                    {"arg1", "val11"},
                    {"arg1", "val12"},
                    {"arg1", "val13"},
                    {"arg2", "val21"},
                    {"arg2", "val22"},
                },
            },
        };

        for (const TCase& c : cases) {
            TDummyReplier replier;
            TSimpleStringInput input("GET /blackbox/?" + c.Params + " HTTP/1.1\n");
            TBufferOutput output;
            THttpInput httpInput(&input);
            THttpOutput httpOutput(&output);
            TRequestReplier::TReplyParams params = {nullptr, httpInput, httpOutput};

            TRequest r(params, replier);

            UNIT_ASSERT_VALUES_EQUAL_C(c.Dups, r.GetDuplicatedArgs(), c.Params);
        }
    }
}
