#include "history.h"

#include <passport/infra/libs/cpp/tail/proto/history.pb.h>

#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/stream/file.h>

namespace NPassport::NTail {
    THistory::THistory(size_t maxHistorySize)
        : MaxHistorySize_(maxHistorySize)
    {
    }

    /**
     * История - монотонно возрастающая очередь записей:
     * ReachedValue_ < Value1 < Value2 < ...
     *
     * История отслеживает лаг по времени, как время жизни первой неотброшенной записи
     */

    bool THistory::AddRecord(ui64 value, TInstant now) {
        if (ReachedValue_ >= value) {
            return ReachedValue_ == value;
        }

        if (!History_.empty()) {
            if (History_.back().Value >= value || History_.back().Timestamp > now) {
                return History_.back().Value == value;
            }

            if (History_.size() == MaxHistorySize_) {
                History_.pop_front();
            }
        }

        History_.emplace_back(TRecord{
            .Timestamp = now,
            .Value = value,
        });

        return true;
    }

    void THistory::DropRecords(ui64 value) {
        ReachedValue_ = std::max(value, ReachedValue_);
        while (!History_.empty() && History_.front().Value <= ReachedValue_) {
            History_.pop_front();
        }
    }

    void THistory::Reset() {
        ReachedValue_ = 0;
        History_.clear();
    }

    TDuration THistory::GetLag(TInstant now) const {
        if (History_.empty() || now < History_.front().Timestamp) {
            return {};
        }

        return now - History_.front().Timestamp;
    }

    bool THistoryCache::Read(const TFsPath& path, ui64 index, THistory& history) {
        history.Reset();

        try {
            TString cache = TFileInput(TFile(path, OpenExisting | RdOnly)).ReadAll();

            history::Cache proto;
            Y_ENSURE(proto.ParseFromString(cache),
                     "failed to parse proto from cache");

            Y_ENSURE(index == proto.index(),
                     "outdated cache: expected " << index << ", got " << proto.index());

            for (const auto& record : proto.history()) {
                Y_ENSURE(history.AddRecord(record.value(), TInstant::Seconds(record.timestamp())),
                         "bad records order");
            }
        } catch (const std::exception& e) {
            TLog::Warning() << "THistoryCache: failed to read history cache from " << path
                            << ": " << e.what();
            return false;
        }

        return true;
    }

    bool THistoryCache::Write(const TFsPath& path, ui64 index, const THistory& history) {
        try {
            google::protobuf::Arena arena;
            history::Cache& proto =
                *google::protobuf::Arena::CreateMessage<history::Cache>(&arena);

            proto.set_index(index);

            auto& records = *proto.mutable_history();
            records.Reserve(history.GetHistory().size());
            for (const auto& record : history.GetHistory()) {
                auto& r = *records.Add();
                r.set_timestamp(record.Timestamp.Seconds());
                r.set_value(record.Value);
            }

            TFileOutput(TFile(path, CreateAlways | WrOnly)).Write(proto.SerializeAsStringOrThrow());
        } catch (const std::exception& e) {
            TLog::Warning() << "THistoryCache: failed to write history cache to " << path
                            << ": " << e.what();
            return false;
        }

        return true;
    }
}
