#include "reader.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/system/fs.h>
#include <util/system/fstat.h>

namespace NPassport::NTail {
    static const TFlags<EOpenModeFlag> FILE_FLAGS = RdOnly | Seq | OpenExisting;

    TFileHolder::TFileHolder(const TFsPath& path)
        : File(path, FILE_FLAGS)
        , Stat(File)
        , Tail(File)
    {
    }

    TReader::TReader(TReaderSettings&& settings)
        : Settings_(std::move(settings))
    {
        Y_ENSURE(Settings_.Path.IsDefined(), "Path argument is missing");
        Y_ENSURE(TryInitFile() || Settings_.MissingPolicy == EMissingPolicy::Ok, "File " << Settings_.Path << " is missing");

        if (File_) {
            size_t position = 0;
            switch (Settings_.BeginningPolicy) {
                case EBeginningPolicy::StartFromEnd:
                    position = File_->Tail.SkipAll();
                    break;
                case EBeginningPolicy::ContinueFromByte:
                    if (File_->Stat.INode != Settings_.INode) {
                        TLog::Warning() << "Inode does not match, starting from top";
                        break;
                    }
                case EBeginningPolicy::StartFromByte:
                    position = File_->Tail.JumpTo(Settings_.Offset);
                    break;
                case EBeginningPolicy::StartFromBeginning:
                    break;
            }
            TLog::Debug() << "Reader: " << Settings_.Path << ": " << position << " bytes skipped";
        }

        if (Settings_.History) {
            History_ = std::make_unique<THistory>(Settings_.History->MaxHistorySize);

            if (File_ && Settings_.History->CachePath.IsDefined()) {
                THistoryCache::Read(Settings_.History->CachePath, File_->Stat.INode, *History_);
            }

            HistoryUpdater_ = std::make_unique<NUtils::TRegularTask>(
                [this]() { this->UpdateHistory(); },
                Settings_.History->UpdatePeriod,
                NUtils::CreateStr("hist-updater-", Settings_.Path.GetPath()));
        }
    }

    const TFsPath& TReader::Path() const {
        return Settings_.Path;
    }

    TDuration TReader::GetLag(TInstant now) {
        std::unique_lock lock(HistoryLock_);

        if (!History_ || !File_) {
            return {};
        }

        History_->DropRecords(File_->Tail.Offset());
        return History_->GetLag(now);
    }

    std::optional<TLine> TReader::ReadLine(TInstant now) {
        if ((!File_ || RotationTimedOut(now)) && !TryInitFile()) {
            return {};
        }

        std::optional<TLine> data = File_->Tail.ReadLine();
        if (data) {
            LastRead_ = now;
            return data;
        }

        if (!LastRotation_) {
            std::optional<TInodePair> iNodes = CheckINode();
            if (!iNodes) {
                return {};
            }

            LastRotation_ = now;
            TLog::Debug() << "Reader: " << Settings_.Path << " inode changed (" << iNodes->Old << " -> " << iNodes->New << ")";
        }

        return {};
    }

    bool TReader::TryInitFile() {
        try {
            std::unique_lock lock(HistoryLock_);

            File_ = std::make_unique<TFileHolder>(Settings_.Path);

            LastRotation_ = TInstant();
            FirstBufferWasSent_ = false;

            if (History_) {
                History_->Reset();
            }

            TLog::Debug() << "Reader: " << Settings_.Path << " opened";
            return true;
        } catch (const TFileError& e) {
            TLog::Debug() << "Reader: " << e.what();
        }

        return false;
    }

    std::optional<TReader::TInodePair> TReader::CheckINode() const {
        Y_ENSURE(File_, "File has not been initialized");
        TInodePair iNodes{
            .Old = File_->Stat.INode,
            .New = TFileStat(Settings_.Path).INode,
        };

        if (iNodes.Old == iNodes.New) {
            return {};
        }
        return iNodes;
    }

    bool TReader::RotationTimedOut(TInstant now) const {
        return LastRotation_ && (now - LastRead_ > Settings_.RotationTimeout || now - LastRotation_ > Settings_.ForceRotationTimeout);
    }

    bool TReader::IsEmptyDataAllowed() const {
        switch (Settings_.EmptyBufferPolicy) {
            case EEmptyBufferPolicy::Never:
                return false;
            case EEmptyBufferPolicy::OnINodeChange:
                return !FirstBufferWasSent_;
        }
    }

    void TReader::UpdateHistory(TInstant now) {
        std::unique_lock lock(HistoryLock_);

        if (History_ && File_) {
            History_->DropRecords(File_->Tail.Offset());
            History_->AddRecord(File_->File.GetLength(), now);

            if (Settings_.History->CachePath.IsDefined()) {
                THistoryCache::Write(Settings_.History->CachePath, File_->Stat.INode, *History_);
            }
        }
    }

    std::optional<TBuffer> TReader::Read(size_t limit) {
        TBuffer buffer;
        buffer.Data.reserve(limit);

        std::optional<TLine> line;
        while ((line = ReadLine())) {
            buffer.Data.append(line->Line);

            if (buffer.Data.size() > limit) {
                break;
            }
        }
        buffer.IsEmpty = buffer.Data.empty();

        if (!File_ || (buffer.IsEmpty && !IsEmptyDataAllowed())) {
            return {};
        }

        FirstBufferWasSent_ = true;
        buffer.INode = File_->Stat.INode;
        buffer.CreateTime = TInstant::Seconds(File_->Stat.CTime);
        buffer.Offset = File_->Tail.Offset() - buffer.Data.size();
        buffer.ROffset = File_->Tail.Offset();

        return buffer;
    }
}
