#pragma once

#include "history.h"
#include "tail.h"

#include <passport/infra/libs/cpp/utils/regular_task.h>

#include <util/datetime/base.h>
#include <util/folder/path.h>

#include <mutex>

namespace NPassport::NTail {
    enum class EMissingPolicy {
        Ok,
        Fail,
    };

    enum class EBeginningPolicy {
        StartFromBeginning, // read all line from the beginning of the file
        StartFromEnd,       // only read lines that appeared after the file was opened
        StartFromByte,      // start reading from specified offset
        ContinueFromByte,   // continue reading the same file from specified offset or read from top, if inode does not match
    };

    enum class EEmptyBufferPolicy {
        Never,
        OnINodeChange,
    };

    struct THistorySettings {
        TFsPath CachePath;
        ui64 MaxHistorySize = 0;
        TDuration UpdatePeriod;
    };

    struct TReaderSettings {
        TFsPath Path;

        EMissingPolicy MissingPolicy = EMissingPolicy::Ok;
        EBeginningPolicy BeginningPolicy = EBeginningPolicy::StartFromEnd;
        EEmptyBufferPolicy EmptyBufferPolicy = EEmptyBufferPolicy::Never;

        ui64 Offset = 0;
        ui64 INode = 0;
        TDuration RotationTimeout;
        TDuration ForceRotationTimeout = TDuration::Minutes(5);

        std::optional<THistorySettings> History;
    };

    // TODO: use TStringBuf and control invalidation in reader
    struct TBuffer {
        size_t INode = 0;
        size_t Offset = 0;
        size_t ROffset = 0;
        TInstant CreateTime; // Last inode modification time, actual creation time is not available on linux

        TString Data;
        bool IsEmpty = true;
    };

    class TFileHolder {
    public:
        explicit TFileHolder(const TFsPath& path);

        const TFile File;
        const TFileStat Stat;

        TTail Tail;
    };

    class TReader {
    public:
        explicit TReader(TReaderSettings&& settings);

        std::optional<TLine> ReadLine(TInstant now = TInstant::Now());

        std::optional<TBuffer> Read(size_t limit);

        const TFsPath& Path() const;

        TDuration GetLag(TInstant now = TInstant::Now());

    private:
        struct TInodePair {
            ui64 Old;
            ui64 New;
        };

        std::optional<TInodePair> CheckINode() const;

        bool RotationTimedOut(TInstant now) const;

        bool TryInitFile();
        bool IsEmptyDataAllowed() const;

        void UpdateHistory(TInstant now = TInstant::Now());

    private:
        const TReaderSettings Settings_;

        TInstant LastRotation_;
        TInstant LastRead_;
        bool FirstBufferWasSent_ = false;

        std::unique_ptr<TFileHolder> File_;

        std::mutex HistoryLock_;
        std::unique_ptr<THistory> History_;
        std::unique_ptr<NUtils::TRegularTask> HistoryUpdater_;
    };
}
