#include "tail.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/datetime/base.h>

namespace NPassport::NTail {
    TTail::TTail(const TFile& file)
        : Mapper_(file)
    {
    }

    size_t TTail::SkipAll() {
        return JumpTo(Mapper_.Length());
    }

    size_t TTail::JumpTo(size_t bytes) {
        PosInFile_ = std::min(i64(bytes), Mapper_.Length());
        return PosInFile_;
    }

    std::optional<TLine> TTail::ReadLine() {
        std::optional<TLine> line;
        do {
            line = TrySplitLine();
        } while (!line && ReadMore());

        return line;
    }

    size_t TTail::Offset() const {
        return ui64(PosInFile_) + PosInMapped_;
    }

    bool TTail::ReadMore() {
        const bool isLengthRefreshingRequired = PosInFile_ + (i64)MapRegion_.MappedSize() == Mapper_.Length();
        PosInFile_ += i64(PosInMapped_);

        const i64 oldLeftSize = i64(MapRegion_.MappedSize()) - i64(PosInMapped_);
        MapRegion_.Reset();
        PosInMapped_ = 0;

        Y_ENSURE(PosInFile_ <= Mapper_.Length(),
                 "posInFile_=" << PosInFile_ << ".size=" << Mapper_.Length());

        if (isLengthRefreshingRequired) {
            Mapper_ = TFileMap(Mapper_.GetFile());
            if (PosInFile_ > Mapper_.Length()) {
                TLog::Error() << "size changed in unexpected way: " << Mapper_.GetFile().GetName()
                              << ". posInFile_=" << PosInFile_ << ". file_size=" << Mapper_.Length()
                              << ". Starting from the very begining";
                PosInFile_ = 0;
            }
        }

        const i64 requiredLength = oldLeftSize + 8192;
        const i64 newLength = std::min(Mapper_.Length() - PosInFile_, requiredLength);

        if (newLength == 0) {
            return false;
        }
        Y_ENSURE(newLength > 0, ""
                                    << "size=" << Mapper_.Length()
                                    << ".posInFile_=" << PosInFile_
                                    << ".requiredLength=" << requiredLength);

        MapRegion_ = Mapper_.Map(PosInFile_, newLength);
        return newLength > oldLeftSize;
    }

    std::optional<TLine> TTail::TrySplitLine() {
        if (!MapRegion_.IsMapped()) {
            return {};
        }
        Y_ENSURE(PosInMapped_ <= MapRegion_.MappedSize());

        TStringBuf buf((const char*)MapRegion_.MappedData() + PosInMapped_,
                       (const char*)MapRegion_.MappedData() + MapRegion_.MappedSize());

        size_t pos = buf.find('\n');
        if (pos == TStringBuf::npos) {
            return {};
        }

        ++pos;
        TLine line{
            .Offset = ui64(PosInFile_) + PosInMapped_,
            .Line = buf.SubStr(0, pos),
        };
        PosInMapped_ += pos;
        return line;
    }
}
