#include <passport/infra/libs/cpp/tail/history.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NTail;

Y_UNIT_TEST_SUITE(History) {
    void CheckHistory(const THistory& history, const std::vector<THistory::TRecord>& records) {
        UNIT_ASSERT_VALUES_EQUAL(records.size(), history.GetHistory().size());
        for (size_t i = 0; i < records.size(); ++i) {
            UNIT_ASSERT_VALUES_EQUAL(records[i].Timestamp, history.GetHistory()[i].Timestamp);
            UNIT_ASSERT_VALUES_EQUAL(records[i].Value, history.GetHistory()[i].Value);
        }
    }

    Y_UNIT_TEST(addRecord) {
        THistory history(3);

        UNIT_ASSERT(history.AddRecord(100, TInstant::Seconds(1644000000)));
        UNIT_ASSERT(history.AddRecord(200, TInstant::Seconds(1644000010)));

        UNIT_ASSERT(!history.AddRecord(300, TInstant::Seconds(1644000000)));
        UNIT_ASSERT(!history.AddRecord(100, TInstant::Seconds(1644000020)));

        UNIT_ASSERT(history.AddRecord(200, TInstant::Seconds(1644000020)));

        CheckHistory(
            history,
            {
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000000), .Value = 100},
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000010), .Value = 200},
            });

        UNIT_ASSERT(history.AddRecord(300, TInstant::Seconds(1644000020)));
        UNIT_ASSERT(history.AddRecord(400, TInstant::Seconds(1644000030)));

        CheckHistory(
            history,
            {
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000010), .Value = 200},
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000020), .Value = 300},
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000030), .Value = 400},
            });
    }

    Y_UNIT_TEST(getLag) {
        TInstant now = TInstant::Seconds(1644849575);

        THistory history(128);

        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(0), history.GetLag(now));

        UNIT_ASSERT(history.AddRecord(100, now - TDuration::Seconds(120)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(120), history.GetLag(now));

        UNIT_ASSERT(history.AddRecord(200, now - TDuration::Seconds(110)));
        UNIT_ASSERT(history.AddRecord(300, now - TDuration::Seconds(100)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(120), history.GetLag(now));

        history.DropRecords(0);
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(120), history.GetLag(now));
        history.DropRecords(99);
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(120), history.GetLag(now));
        history.DropRecords(100);
        UNIT_ASSERT(history.AddRecord(350, now - TDuration::Seconds(95)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(110), history.GetLag(now));
        history.DropRecords(500);
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(0), history.GetLag(now));

        UNIT_ASSERT(!history.AddRecord(400, now - TDuration::Seconds(90)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(0), history.GetLag(now));
        UNIT_ASSERT(history.AddRecord(500, now - TDuration::Seconds(80)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(0), history.GetLag(now));
        UNIT_ASSERT(history.AddRecord(501, now - TDuration::Seconds(70)));
        UNIT_ASSERT_VALUES_EQUAL(TDuration::Seconds(70), history.GetLag(now));
    }

    Y_UNIT_TEST(reset) {
        THistory history(128);

        history.AddRecord(10);
        history.AddRecord(20);

        history.Reset();
        UNIT_ASSERT_VALUES_EQUAL(0, history.GetHistory().size());

        history.DropRecords(10);
        history.Reset();
        history.AddRecord(5);
        UNIT_ASSERT_VALUES_EQUAL(1, history.GetHistory().size());
    }

    Y_UNIT_TEST(cache) {
        TFsPath path = GetOutputPath() / "test_history_cache";

        THistory history(1024);

        TFileOutput(path).Write("some random text");
        UNIT_ASSERT(!THistoryCache::Read(path, 0, history));
        UNIT_ASSERT_VALUES_EQUAL(0, history.GetHistory().size());

        NFs::Remove(path);
        UNIT_ASSERT(!THistoryCache::Read(path, 0, history));
        UNIT_ASSERT_VALUES_EQUAL(0, history.GetHistory().size());

        UNIT_ASSERT(THistoryCache::Write(path, 129, history));
        UNIT_ASSERT(THistoryCache::Read(path, 129, history));
        UNIT_ASSERT_VALUES_EQUAL(0, history.GetHistory().size());

        history.AddRecord(100, TInstant::Seconds(1644000000));
        history.AddRecord(200, TInstant::Seconds(1644000100));
        UNIT_ASSERT(THistoryCache::Write(path, 100500, history));
        history.Reset();
        UNIT_ASSERT(!THistoryCache::Read(path, 129, history));
        CheckHistory(history, {});
        UNIT_ASSERT(THistoryCache::Read(path, 100500, history));
        CheckHistory(
            history,
            {
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000000), .Value = 100},
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000100), .Value = 200},
            });

        history.Reset();
        history.AddRecord(150, TInstant::Seconds(1644000050));
        UNIT_ASSERT(THistoryCache::Write(path, 100501, history));
        history.Reset();
        UNIT_ASSERT(THistoryCache::Read(path, 100501, history));
        CheckHistory(
            history,
            {
                THistory::TRecord{.Timestamp = TInstant::Seconds(1644000050), .Value = 150},
            });
    }
}
