#include <passport/infra/libs/cpp/tail/reader.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NTail;

Y_UNIT_TEST_SUITE(Reader) {
    const TString filename("ololo.log");
    const TString someString("asddsadasdasdasddsadasdasdasddsadasdasdasddsadasdasdasddsadasda78");
    const size_t batchSize = 1000;

    void PopulateFile(TFileOutput& out) {
        for (size_t i = 0; i < batchSize; ++i) {
            out << someString << '\n';
        }
        out << Flush;
    }

    Y_UNIT_TEST(ReadFromTop) {
        NFs::Remove(filename);
        TFileOutput out(filename);

        PopulateFile(out);

        std::vector<TLine> output;
        TReader reader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Fail,
            .BeginningPolicy = EBeginningPolicy::StartFromBeginning,
        });
        std::optional<TLine> data;
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(batchSize, output.size());

        NFs::Remove(filename);
        TFileOutput out2(filename);
        PopulateFile(out2);

        UNIT_ASSERT(!reader.ReadLine());
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(batchSize * 2, output.size());
    }

    Y_UNIT_TEST(ReadFromBottom) {
        NFs::Remove(filename);
        TFileOutput out(filename);

        PopulateFile(out);

        std::vector<TLine> output;
        TReader reader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Fail,
            .BeginningPolicy = EBeginningPolicy::StartFromEnd,
        });
        std::optional<TLine> data;
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(0, output.size());

        PopulateFile(out);
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(batchSize, output.size());

        NFs::Remove(filename);
        TFileOutput out2(filename);
        PopulateFile(out2);

        UNIT_ASSERT(!reader.ReadLine());
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(batchSize * 2, output.size());
    }

    Y_UNIT_TEST(ReadFromByte) {
        NFs::Remove(filename);
        TFileOutput out(filename);

        PopulateFile(out);

        ui64 halfBatchSize = batchSize / 2;

        std::vector<TLine> output;
        TReader reader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Fail,
            .BeginningPolicy = EBeginningPolicy::StartFromByte,

            .Offset = halfBatchSize * (someString.size() + 1),
            .INode = TFileStat(filename).INode,
        });
        std::optional<TLine> data;
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(halfBatchSize, output.size());

        PopulateFile(out);
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(halfBatchSize + batchSize, output.size());

        NFs::Remove(filename);
        TFileOutput out2(filename);
        PopulateFile(out2);

        UNIT_ASSERT(!reader.ReadLine());
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(halfBatchSize + batchSize * 2, output.size());
    }

    Y_UNIT_TEST(ContinueFromByte) {
        NFs::Remove(filename);
        TFileOutput out(filename);
        PopulateFile(out);

        std::vector<TLine> output;
        TReader reader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Fail,
            .BeginningPolicy = EBeginningPolicy::ContinueFromByte,

            .Offset = batchSize,
            .INode = 0,
        });
        std::optional<TLine> data;
        while ((data = reader.ReadLine())) {
            output.emplace_back(*data);
        }
        UNIT_ASSERT_VALUES_EQUAL(batchSize, output.size());
    }

    Y_UNIT_TEST(Missing) {
        NFs::Remove(filename);
        UNIT_ASSERT_NO_EXCEPTION(TReader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Ok,
        }));

        TReader reader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Ok,
        });
        UNIT_ASSERT_NO_EXCEPTION(reader.ReadLine());

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TReader(TReaderSettings{
                .Path = filename,
                .MissingPolicy = EMissingPolicy::Fail,
            }), yexception, NUtils::CreateStr("File ", filename, " is missing"));
    }

    Y_UNIT_TEST(WaitForData) {
        NFs::Remove(filename);
        TFileOutput out1(filename);
        TReader reader(TReaderSettings{
            .Path = filename,
            .MissingPolicy = EMissingPolicy::Fail,
            .BeginningPolicy = EBeginningPolicy::StartFromEnd,
            .RotationTimeout = TDuration::Seconds(1),
            .ForceRotationTimeout = TDuration::Seconds(2),
        });

        TString data1 = NUtils::CreateStr("1: ", someString, '\n');
        TString data2 = NUtils::CreateStr("2: ", someString, '\n');
        TString data3 = NUtils::CreateStr("3: ", someString, '\n');

        out1 << data1 << Flush;
        std::optional<TLine> line = reader.ReadLine();
        UNIT_ASSERT(line);
        UNIT_ASSERT_VALUES_EQUAL(data1, line->Line);

        NFs::Rename(filename, NUtils::CreateStr(filename, ".1"));
        TFileOutput out2(filename);

        out2 << data2 << Flush;
        UNIT_ASSERT(!reader.ReadLine());

        TInstant now = TInstant::Now();
        for (size_t i = 0; i < 15; ++i) {
            out1 << data1 << Flush;
            line = reader.ReadLine(now);
            UNIT_ASSERT(line);
            UNIT_ASSERT_VALUES_EQUAL(data1, line->Line);
            now += TDuration::MilliSeconds(100);
        }

        now += TDuration::MilliSeconds(500);
        line = reader.ReadLine(now);
        UNIT_ASSERT(line);
        UNIT_ASSERT_VALUES_EQUAL(data2, line->Line);
        UNIT_ASSERT_VALUES_EQUAL(0, line->Offset);

        out1 << data1 << Flush;
        UNIT_ASSERT(!reader.ReadLine(now));

        NFs::Rename(filename, NUtils::CreateStr(filename, ".2"));
        TFileOutput out3(filename);
        out3 << data3 << Flush;

        UNIT_ASSERT(!reader.ReadLine(now));
        now += TDuration::MilliSeconds(1100);
        out2 << data2 << Flush;

        line = reader.ReadLine(now);
        UNIT_ASSERT(line);
        UNIT_ASSERT_VALUES_EQUAL(data3, line->Line);
        UNIT_ASSERT_VALUES_EQUAL(0, line->Offset);
    }

    Y_UNIT_TEST(ReadFromEmptyFile) {
        NFs::Remove(filename);

        TReaderSettings settings{
            .Path = filename,
            .EmptyBufferPolicy = EEmptyBufferPolicy::Never,
            .Offset = batchSize,
        };

        TReader readerWithoutEmpty = TReader(TReaderSettings(settings));
        settings.EmptyBufferPolicy = EEmptyBufferPolicy::OnINodeChange;
        TReader readerWithEmpty = TReader(TReaderSettings(settings));

        std::optional<NTail::TBuffer> data;
        UNIT_ASSERT(!readerWithoutEmpty.Read(100));
        UNIT_ASSERT(!readerWithEmpty.Read(100));

        TFileOutput out(filename);
        UNIT_ASSERT(!readerWithoutEmpty.Read(100));

        UNIT_ASSERT(readerWithEmpty.Read(100));
        UNIT_ASSERT(!readerWithEmpty.Read(100));
    }
}
