#include <passport/infra/libs/cpp/tail/tail.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NTail;

template <>
void Out<TLine>(IOutputStream& o, const TLine& value) {
    o << value.Offset << ": " << value.Line;
}

Y_UNIT_TEST_SUITE(Tail) {
    const TString filename("ololo.log");
    const TString someString("asddsadasdasdasddsadasdasdasddsadasdasdasddsadasdasdasddsadasda78");
    const size_t batchSize = 1000;

    Y_UNIT_TEST(Simple) {
        NFs::Remove(filename);
        TFileOutput out(filename);
        TFile file(filename, RdOnly);
        TTail reader(file);

        std::optional<TLine> res = reader.ReadLine();
        UNIT_ASSERT(!res);
        UNIT_ASSERT_VALUES_EQUAL(0, reader.Offset());

        out << someString << Flush;
        res = reader.ReadLine();
        UNIT_ASSERT(!res);
        UNIT_ASSERT_VALUES_EQUAL(0, reader.Offset());

        out << Endl;
        res = reader.ReadLine();
        UNIT_ASSERT(res);
        UNIT_ASSERT(res->Line.ChopSuffix("\n"));
        UNIT_ASSERT_STRINGS_EQUAL(someString, res->Line);
        UNIT_ASSERT_VALUES_EQUAL(0, res->Offset);

        res = reader.ReadLine();
        UNIT_ASSERT(!res);
        UNIT_ASSERT_VALUES_EQUAL(66, reader.Offset());
    }

    Y_UNIT_TEST(ManyLines) {
        NFs::Remove(filename);
        TFileOutput out(filename);
        TFile file(filename, RdOnly);
        TTail reader(file);

        for (size_t i = 0; i < batchSize; ++i) { // buffer is about 8192
            out << someString << '\n';
        }
        out << Flush;

        {
            int idx = 0;
            std::optional<TLine> res = reader.ReadLine();
            while (res) {
                UNIT_ASSERT_VALUES_EQUAL(idx * (someString.size() + 1), res->Offset);
                UNIT_ASSERT(res->Line.ChopSuffix("\n"));
                UNIT_ASSERT_VALUES_EQUAL_C(someString, res->Line, ". idx=" << idx);
                ++idx;
                res = reader.ReadLine();
            }
            UNIT_ASSERT_VALUES_EQUAL(idx, batchSize);
        }
    }

    Y_UNIT_TEST(LongLines) {
        NFs::Remove(filename);
        TFileOutput out(filename);
        TFile file(filename, RdOnly);
        TTail reader(file);

        TStringBuf data(someString);

        out << data.SubStr(0, 7) << Flush;
        std::optional<TLine> res = reader.ReadLine();
        UNIT_ASSERT(!res);
        UNIT_ASSERT_VALUES_EQUAL(0, reader.Offset());

        out << data.SubStr(7) << Flush;
        res = reader.ReadLine();
        UNIT_ASSERT(!res);
        UNIT_ASSERT_VALUES_EQUAL(0, reader.Offset());

        out << Endl;
        res = reader.ReadLine();
        UNIT_ASSERT(res);
        UNIT_ASSERT(res->Line.ChopSuffix("\n"));
        UNIT_ASSERT_STRINGS_EQUAL(data, res->Line);
        UNIT_ASSERT_VALUES_EQUAL(0, res->Offset);

        for (size_t i = 0; i < batchSize; ++i) { // buffer is about 8192
            out << data;
        }
        out << Flush;
        res = reader.ReadLine();
        UNIT_ASSERT(!res);
        UNIT_ASSERT_VALUES_EQUAL(66, reader.Offset());

        out << Endl;
        res = reader.ReadLine();
        UNIT_ASSERT(res);
        UNIT_ASSERT_VALUES_EQUAL(66, res->Offset);
        UNIT_ASSERT(res->Line.ChopSuffix("\n"));
        UNIT_ASSERT_VALUES_EQUAL(1000 * data.size(), res->Line.size());
    }
}
