#include "private_key.h"

#include "service_tickets.h"

#include <passport/infra/libs/cpp/tvm/signer/signer.h>

#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/tvmauth/checked_service_ticket.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/tvmauth/client/misc/disk_cache.h>
#include <library/cpp/tvmauth/deprecated/user_context.h>
#include <library/cpp/tvmauth/src/parser.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

namespace NPassport::NTvmCommon {
    static TString GetPublicKeysFromCache(const TFsPath& file) {
        NTvmAuth::TDiskReader reader(file);
        if (!reader.Read()) {
            throw yexception() << "Failed to read cache from " << file;
        }
        return reader.Data();
    }

    TPrivateKey::TPrivateKey(TPrivateKeySettings&& settings,
                             std::shared_ptr<NTvmAuth::TTvmClient> tvmClient)
        : Settings_(std::move(settings))
        , PublicKeysCacheFile_(TFsPath(Settings_.TvmCacheDir) / "public_keys")
        , PrivateKeysCacheFile_(TFsPath(Settings_.TvmCacheDir) / "private_keys")
        , TvmClient_(std::move(tvmClient))
    {
        Y_ENSURE(TvmClient_, "Tvm client has not been initialized");
        LastUpdateTs_.Set(std::make_shared<TInstant>());
        LastErrorMessage_.Set(std::make_shared<TString>());

        try {
            if (Settings_.EnableDiskCache) {
                UpdateKeys(FetchFromCache());
            } else {
                TLog::Debug("Private key cache is disabled");
            }
        } catch (const std::exception& e) {
            TLog::Debug("Failed to load private keys from cache: %s", e.what());
        }

        try {
            Update();
        } catch (const std::exception& e) {
            throw yexception() << "Failed to read private keys from TVM API: " << e.what();
        }
    }

    void TPrivateKey::Run() {
        try {
            Update();
        } catch (const std::exception& e) {
            LastErrorMessage_.Set(std::make_shared<TString>(e.what()));
        }
    }

    TPrivateKey::TKey TPrivateKey::GetKey() const {
        return Key_.Get();
    }

    void TPrivateKey::SelfCheck(time_t now) const {
        Y_ENSURE(Settings_.EnableDiskCache, "Cannot perform self-check, when cache is disabled");

        NTicketSigner::TUserSigner s;
        s.AddUid(123);           // dummy
        s.SetDefaultUid(123);    // dummy
        s.SetEntryPoint(100500); // dummy
        s.SetEnv(Settings_.Env);
        TKey k = GetKey();
        const TString t = s.SerializeV3(*k, now);

        NTvmAuth::TUserContext ctx(Settings_.Env, GetPublicKeysFromCache(PublicKeysCacheFile_));
        NTvmAuth::TCheckedUserTicket u = ctx.Check(t);
        if (!u) {
            throw yexception() << "Basic check is failed: " << NTvmAuth::StatusToString(u.GetStatus());
        }
    }

    NJuggler::TStatus TPrivateKey::GetJugglerStatus() const {
        TDuration elapsed = TInstant::Now() - *LastUpdateTs_.Get();

        auto createMsg = [this, &elapsed]() {
            return NUtils::CreateStr("Private keys have not been updated from TVM API for ",
                                     elapsed.Seconds(), " seconds: ", *LastErrorMessage_.Get());
        };

        if (elapsed > Settings_.JugglerCriticalTimeout) {
            return NJuggler::TStatus(NJuggler::ECode::Critical, createMsg());
        }

        if (elapsed > Settings_.JugglerWarningTimeout) {
            return NJuggler::TStatus(NJuggler::ECode::Warning, createMsg());
        }

        return {};
    }

    void TPrivateKey::ChooseKey(std::vector<TKeyId>&& keyIds,
                                size_t preferredKeyIdx,
                                std::function<bool(const TKeyId id)> func) {
        std::vector<TKeyId> sorted = std::move(keyIds);
        std::sort(sorted.begin(), sorted.end(), [](TKeyId l, TKeyId r) -> bool {
            // Reverse order of keys
            // Key generating script can successfully add new key and fail to delete old
            // So actual key should be correctly changed
            return r < l;
        });

        if (sorted.empty()) {
            throw yexception() << "No private key for work";
        }

        preferredKeyIdx = preferredKeyIdx > 0 ? preferredKeyIdx - 1 : 0;
        if (sorted.size() <= preferredKeyIdx) {
            preferredKeyIdx = sorted.size() == 1 ? 0 : sorted.size() - 1;
        }

        for (size_t i = preferredKeyIdx; i < sorted.size(); ++i) {
            if (func(sorted.at(i))) {
                return;
            }
        }

        for (int i = preferredKeyIdx - 1; i >= 0; --i) {
            if (func(sorted.at(i))) {
                return;
            }
        }

        throw yexception() << "All Tvm keys are malformed. Impossible to work";
    }

    NTvmAuth::EBlackboxEnv TPrivateKey::TranslateEnv(const TString& name) {
        if (name == "prod") {
            return NTvmAuth::EBlackboxEnv::Prod;
        }
        if (name == "prod_yateam") {
            return NTvmAuth::EBlackboxEnv::ProdYateam;
        }
        if (name == "test") {
            return NTvmAuth::EBlackboxEnv::Test;
        }
        if (name == "test_yateam") {
            return NTvmAuth::EBlackboxEnv::TestYateam;
        }
        if (name == "stress") {
            return NTvmAuth::EBlackboxEnv::Stress;
        }
        throw yexception() << "Supported envs: prod, prod_yateam, test, test_yateam, stress. Env is unknown: "
                           << name;
    }

    void TPrivateKey::Update() {
        if (TInstant::Now() - *LastUpdateTs_.Get() < Settings_.Period) {
            return;
        }

        TString serviceTicket = TvmClient_->GetServiceTicketFor(NTvmCommon::TServiceTickets::TVMAPI_);
        UpdateKeys(FetchFromApi(serviceTicket));
    }

    void TPrivateKey::UpdateKeys(const TStampedPrivateKey& key) {
        Key_.Set(Parse(key.Data));

        if (Settings_.EnableDiskCache) {
            NTvmAuth::TDiskWriter writer(PrivateKeysCacheFile_);
            if (!writer.Write(key.Data, key.Ts)) {
                LastErrorMessage_.Set(std::make_shared<TString>("Failed to write private keys to cache"));
                TLog::Warning() << "Failed to write private keys to cache: " << *LastErrorMessage_.Get();
                return;
            }
        }

        LastUpdateTs_.Set(std::make_shared<TInstant>(key.Ts));
        LastErrorMessage_.Set(std::make_shared<TString>());
        TLog::Info() << "PrivateKey: new private keys loaded with ts: " << *LastUpdateTs_.Get();
    }

    TPrivateKey::TKey TPrivateKey::Parse(const TStringBuf privateKey) const {
        TString protoArr = NTvmAuth::TParserTvmKeys::ParseStrV1(privateKey);
        tvm_keys::Keys proto;
        if (!proto.ParseFromString(protoArr)) {
            throw yexception() << "Failed to parse protobuf with private keys from TVM";
        }

        std::vector<TKeyId> keyIds;
        keyIds.reserve(proto.bb_size());
        for (int idx = 0; idx < proto.bb_size(); ++idx) {
            keyIds.push_back(proto.bb(idx).gen().id());
        }

        TKey key;
        TPrivateKey::ChooseKey(
            std::move(keyIds),
            Settings_.PreferredKeyIdx,
            [&proto, &key, this](const TKeyId id) -> bool {
                try {
                    for (int idx = 0; idx < proto.bb_size(); ++idx) {
                        if (id != proto.bb(idx).gen().id()) {
                            continue;
                        }

                        if (proto.bb(idx).env() != tvm_keys::BbEnvType(int(Settings_.Env))) {
                            TLog::Error("Failed to add private key. Wrong env: %d. Id %d",
                                        int(proto.bb(idx).env()),
                                        id);
                            continue;
                        }

                        key = std::make_shared<NTvmAuth::NRw::TRwPrivateKey>(
                            proto.bb(idx).gen().body(),
                            id);
                        return true;
                    }
                } catch (const std::exception& e) {
                    TLog::Error("Failed to load private key: %s", e.what());
                }

                return false;
            });

        return key;
    }

    const static TString X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";
    const static TString TVM_PRIVATE_KEY_PATH = "/2/private_keys";

    TPrivateKey::TStampedPrivateKey TPrivateKey::FetchFromApi(const TString& serviceTicket) const {
        TKeepAliveHttpClient httpClient(Settings_.TvmHost, Settings_.TvmPort);

        TKeepAliveHttpClient::THeaders headers{
            {X_YA_SERVICE_TICKET, serviceTicket},
        };
        TStringStream response;

        TKeepAliveHttpClient::THttpCode status = httpClient.DoPost(TVM_PRIVATE_KEY_PATH, "", &response, headers);
        Y_ENSURE(status == HTTP_OK, NUtils::CreateStr("Request failed: ", response.Str()));

        return {
            .Ts = TInstant::Now(),
            .Data = response.Str(),
        };
    }

    TPrivateKey::TStampedPrivateKey TPrivateKey::FetchFromCache() const {
        NTvmAuth::TDiskReader reader(PrivateKeysCacheFile_);
        Y_ENSURE(reader.Read(), "Failed to read private keys from cache");
        Y_ENSURE(TInstant::Now() - reader.Time() < Settings_.CacheTtl, "Cache is too old");

        return {
            .Ts = reader.Time(),
            .Data = reader.Data(),
        };
    }
}
