#include "public_keys.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <utility>

namespace NPassport::NTvmCommon {
    TPublicKeys::TPublicKeys(const TString& keyfile,
                             TDuration period,
                             NTvmAuth::TTvmId selfClientId,
                             NTvmAuth::EBlackboxEnv env)
        : File_(keyfile)
        , SelfClientId_(selfClientId)
        , Env_(env)
    {
        Loader_ = std::make_unique<NUtils::TFileLoader>(
            File_,
            [this](const TStringBuf fileBody, const time_t) { Update(fileBody); },
            period);
    }

    TPublicKeys::TPublicKeys(const TString& keyfile,
                             TDuration period,
                             NTvmAuth::TTvmId selfClientId)
        : File_(keyfile)
        , SelfClientId_(selfClientId)
    {
        Loader_ = std::make_unique<NUtils::TFileLoader>(
            File_,
            [this](const TStringBuf fileBody, const time_t) { Update(fileBody); },
            period);
    }

    TPublicKeys::~TPublicKeys() = default;

    NTvmAuth::TCheckedServiceTicket TPublicKeys::CheckServiceTicket(const TStringBuf ticket) const {
        TCtx ctx = Ctx_.Get();
        NTvmAuth::TCheckedServiceTicket t = ctx->Serv.Check(ticket);
        if (!t) {
            TStringBuf okToLogTicket = NTvmAuth::NUtils::RemoveTicketSignature(ticket);
            TLog::Debug("Failed to check service ticket: %s. Body: %s",
                        NTvmAuth::StatusToString(t.GetStatus()).data(),
                        TString(okToLogTicket.data(), okToLogTicket.size()).c_str());
        }
        return t;
    }

    NTvmAuth::TCheckedUserTicket TPublicKeys::CheckUserTicket(const TString& ticket) const {
        if (!Env_) {
            throw yexception() << "Enviroment for blackbox was not set";
        }

        TCtx ctx = Ctx_.Get();
        NTvmAuth::TCheckedUserTicket t = ctx->User->Check(ticket);
        if (!t) {
            TStringBuf okToLogTicket = NTvmAuth::NUtils::RemoveTicketSignature(ticket);
            TLog::Debug("Failed to check user ticket: %s. Body: %s",
                        NTvmAuth::StatusToString(t.GetStatus()).data(),
                        TString(okToLogTicket.data(), okToLogTicket.size()).c_str());
        }
        return t;
    }

    NTvmAuth::TTvmId TPublicKeys::GetSelfClientId() const {
        return SelfClientId_;
    }

    void TPublicKeys::Update(const TStringBuf fileBody) {
        Ctx_.Set(Parse(fileBody));
        TLog::Info("PublicKeys: new public keys fetched from file: %s", File_.c_str());
    }

    TPublicKeys::TCtx TPublicKeys::Parse(const TStringBuf fileBody) const {
        return std::make_shared<TContexts>(
            TContexts{
                NTvmAuth::TServiceContext::CheckingFactory(SelfClientId_, fileBody),
                Env_ ? TOptUserCtx(std::in_place_t{}, *Env_, fileBody)
                     : TOptUserCtx()});
    }
}
