#include "service_tickets.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NTvmCommon {
    const TString TServiceTickets::BLACKBOX_ = "blackbox";
    const TString TServiceTickets::ISSNKMS_ = "issnkms";
    const TString TServiceTickets::KOLMOGOR_ = "kolmogor";
    const TString TServiceTickets::LOGBROKER_ = "logbroker";
    const TString TServiceTickets::TVMAPI_ = "tvmapi";
    const TString TServiceTickets::PASSPORTAPI_ = "passportapi";
    const TString TServiceTickets::STAFFAPI_ = "staffapi";

    TServiceTickets::TServiceTickets(const TString& filename, const TDsts& dst, TDuration period)
        : Filename_(filename)
        , Dst_(dst)
    {
        Refresher_ = std::make_unique<NUtils::TFileLoader>(
            filename,
            [this](const TStringBuf fileBody, time_t mtime) {
                Update(fileBody);
                Mtime_.store(TInstant::Seconds(mtime));
            },
            period);
    }

    TServiceTickets::~TServiceTickets() = default;

    TString TServiceTickets::GetServiceTicket(const TString& dst) const {
        const TTicketsPtr all = ServiceTickets_.Get();

        auto it = all->find(dst);
        if (it == all->end()) {
            throw yexception() << "Cannot find service ticket for dst: " << dst;
        }

        return it->second;
    }

    bool TServiceTickets::IsOk(TString& out) const {
        out.clear();

        const TInstant lastSuccess = Mtime_.load(std::memory_order_relaxed);
        const TInstant deadline =
            lastSuccess +
            TDuration::Hours(12) - // service ticket ttl
            TDuration::Minutes(1); // to protect from lag of L3 balancer ping

        if (deadline < TInstant::Now()) {
            out = TStringBuilder() << "Service tickets are expired (must be refreshed every 1 hour), last success was: "
                                   << lastSuccess;
            return false;
        }

        return true;
    }

    void TServiceTickets::Update(const TStringBuf fileBody) {
        ServiceTickets_.Set(Parse(fileBody));
        TLog::Info("ServiceTickets: new ServiceTickets fetched from file: %s", Filename_.c_str());
    }

    TServiceTickets::TTicketsPtr TServiceTickets::Parse(const TStringBuf fileBody) const {
        rapidjson::Document doc;
        Y_ENSURE(NJson::TReader::DocumentAsObject(fileBody, doc),
                 "Failed to parse json with service tickets: " << fileBody);

        const rapidjson::Value* tickets = nullptr;
        Y_ENSURE(NJson::TReader::MemberAsObject(doc, "tickets", tickets),
                 "Failed to get 'tickets': " << fileBody);

        TTicketsPtr res = std::make_shared<TTickets>();

        for (auto it = tickets->MemberBegin(); it != tickets->MemberEnd(); ++it) {
            const rapidjson::Value& jsonObj = it->value;
            if (!jsonObj.IsObject()) {
                TLog::Error() << "Failed to parse TVM tickets (object): " << fileBody;
                continue;
            }

            TString err;
            if (NJson::TReader::MemberAsString(jsonObj, "error", err)) {
                TLog::Error("Failed to get ticket for dst '%s': %s", it->name.GetString(), err.c_str());
                continue;
            }

            TString alias;
            TString ticket;
            if (!NJson::TReader::MemberAsString(jsonObj, "ticket", ticket) ||
                !NJson::TReader::MemberAsString(jsonObj, "alias", alias)) {
                TLog::Error() << "Invalid format from tvm-keyring (no ticket or no alias): " << fileBody;
                continue;
            }

            res->emplace(std::move(alias), std::move(ticket));
        }

        CheckDsts(*res);

        return res;
    }

    void TServiceTickets::CheckDsts(const TServiceTickets::TTickets& t) const {
        for (const TString& d : Dst_) {
            if (t.find(d) == t.end()) {
                throw yexception() << "Required service ticket is absent: " << d;
            }
        }
    }

}
