#include <passport/infra/libs/cpp/tvm/common/private_key.h>

#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/utils/file.h>

#include <library/cpp/http/server/response.h>
#include <library/cpp/testing/mock_server/server.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/tvmauth/client/mocked_updater.h>
#include <library/cpp/tvmauth/client/misc/disk_cache.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NTvmCommon;

template <>
void Out<NJuggler::TStatus>(IOutputStream& o, const NJuggler::TStatus& value) {
    o << "status: (" << value.StatusCode() << ", " << value.Message() << ")";
}

Y_UNIT_TEST_SUITE(TvmCommonPrivate) {
    TString GetFile(TStringBuf name) {
        return (ArcadiaSourceRoot() + "/passport/infra/libs/cpp/tvm/common/ut/" + name).data();
    }

    class TGetPrivateKeys: public TRequestReplier {
    public:
        TGetPrivateKeys() = default;

        bool DoReply(const TReplyParams& params) override {
            TStringBuf path = TParsedHttpFull(params.Input.FirstLine()).Path;
            TString data = params.Input.ReadAll();
            THttpHeaders headers = params.Input.Headers();
            THttpResponse resp;

            if (path != "/2/private_keys") {
                resp.SetHttpCode(HTTP_NOT_FOUND);
                resp.OutTo(params.Output);
                return true;
            }

            if (!headers.HasHeader("X-Ya-Service-Ticket")) {
                resp.SetHttpCode(HTTP_FORBIDDEN);
                resp.OutTo(params.Output);
                return true;
            }

            TString filename = headers.FindHeader("X-Ya-Service-Ticket")->Value();

            resp.SetHttpCode(HTTP_OK);
            try {
                resp.SetContent(NUtils::ReadFile(GetFile(filename)));
            } catch (const std::exception& e) {
                resp.SetContent("malformed key");
            }
            resp.OutTo(params.Output);

            return true;
        }
    };

    static std::shared_ptr<NTvmAuth::TTvmClient> CreateMockedUpdater(const TString& st) {
        return std::make_shared<NTvmAuth::TTvmClient>(
            MakeIntrusiveConst<NTvmAuth::TMockedUpdater>(
                NTvmAuth::TMockedUpdater::TSettings{
                    .SelfTvmId = 224,
                    .Backends = {
                        {"tvmapi", 172, st},
                    },
                }));
    }

    void PrepareCache(TInstant t = TInstant::Now()) {
        TString okPrivateKey = NUtils::ReadFile(GetFile("ok.private"));
        TString okPublicKey = NUtils::ReadFile(GetFile("ok.public"));
        TString badPrivateKey = NUtils::ReadFile(GetFile("bad.private"));
        TString badPublicKey = NUtils::ReadFile(GetFile("bad.public"));

        NFs::MakeDirectory(GetOutputPath() / "bad");
        NFs::MakeDirectory(GetOutputPath() / "ok");

        {
            NTvmAuth::TDiskWriter writer(GetOutputPath() / "bad" / "private_keys");
            writer.Write(badPrivateKey, t);
        }
        {
            NTvmAuth::TDiskWriter writer(GetOutputPath() / "bad" / "public_keys");
            writer.Write(badPublicKey, t);
        }
        {
            NTvmAuth::TDiskWriter writer(GetOutputPath() / "ok" / "private_keys");
            writer.Write(okPrivateKey, t);
        }
        {
            NTvmAuth::TDiskWriter writer(GetOutputPath() / "ok" / "public_keys");
            writer.Write(okPublicKey, t);
        }
    }

    Y_UNIT_TEST(chooseKey) {
        UNIT_ASSERT_EXCEPTION(TPrivateKey::ChooseKey({}, 3, [](const TKeyId) { return false; }), yexception);
        UNIT_ASSERT_EXCEPTION(TPrivateKey::ChooseKey(
                                  {1, 2, 3, 4, 5, 6, 7},
                                  3,
                                  [](const TKeyId) { return false; }),
                              yexception);

        ui32 res;
        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            3,
            [&res](const TKeyId id) { res = id; return true; });
        UNIT_ASSERT_EQUAL(5, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            7,
            [&res](const TKeyId id) { res = id; return true; });
        UNIT_ASSERT_EQUAL(1, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            1,
            [&res](const TKeyId id) { res = id; return true; });
        UNIT_ASSERT_EQUAL(7, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            3,
            [&res](const TKeyId id) { res = id; return id == 5; });
        UNIT_ASSERT_EQUAL(5, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            3,
            [&res](const TKeyId id) { res = id; return id == 1; });
        UNIT_ASSERT_EQUAL(1, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            3,
            [&res](const TKeyId id) { res = id; return id == 7; });
        UNIT_ASSERT_EQUAL(7, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            3,
            [&res](const TKeyId id) { res = id; return id == 1 || id == 7; });
        UNIT_ASSERT_EQUAL(1, res);

        TPrivateKey::ChooseKey(
            {1, 2, 3, 4, 5, 6, 7},
            0,
            [&res](const TKeyId id) { res = id; return id == 1 || id == 7; });
        UNIT_ASSERT_VALUES_EQUAL(7, res);

        TPrivateKey::ChooseKey(
            {1},
            3,
            [&res](const TKeyId id) { res = id; return id == 1; });
        UNIT_ASSERT_VALUES_EQUAL(1, res);
    }

    Y_UNIT_TEST(translateEnv) {
        UNIT_ASSERT_EQUAL(NTvmAuth::EBlackboxEnv::Prod, TPrivateKey::TranslateEnv("prod"));
        UNIT_ASSERT_EQUAL(NTvmAuth::EBlackboxEnv::ProdYateam, TPrivateKey::TranslateEnv("prod_yateam"));
        UNIT_ASSERT_EQUAL(NTvmAuth::EBlackboxEnv::Test, TPrivateKey::TranslateEnv("test"));
        UNIT_ASSERT_EQUAL(NTvmAuth::EBlackboxEnv::TestYateam, TPrivateKey::TranslateEnv("test_yateam"));
        UNIT_ASSERT_EQUAL(NTvmAuth::EBlackboxEnv::Stress, TPrivateKey::TranslateEnv("stress"));
        UNIT_ASSERT_EXCEPTION(TPrivateKey::TranslateEnv("stress "), yexception);
        UNIT_ASSERT_EXCEPTION(TPrivateKey::TranslateEnv(" stress"), yexception);
        UNIT_ASSERT_EXCEPTION(TPrivateKey::TranslateEnv("stres"), yexception);
        UNIT_ASSERT_EXCEPTION(TPrivateKey::TranslateEnv("production"), yexception);
    }

    Y_UNIT_TEST(InitializationWithCache) {
        PrepareCache();

        std::shared_ptr<NTvmAuth::TTvmClient> tvmClient = CreateMockedUpdater("asdf");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TPrivateKey(
                                           TPrivateKeySettings{
                                               .TvmHost = "",
                                           },
                                           {}),
                                       yexception, "Tvm client has not been initialized");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TPrivateKey(
                                           TPrivateKeySettings{
                                               .TvmHost = "",
                                               .TvmCacheDir = GetOutputPath() / "missing",
                                               .Env = NTvmAuth::EBlackboxEnv::Prod,
                                           },
                                           tvmClient),
                                       yexception, "Failed to read private keys from TVM API: ");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TPrivateKey(
                                           TPrivateKeySettings{
                                               .TvmHost = "",
                                               .TvmCacheDir = GetOutputPath() / "bad",
                                               .Env = NTvmAuth::EBlackboxEnv::Prod,
                                           },
                                           tvmClient),
                                       yexception, "Failed to read private keys from TVM API: ");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TPrivateKey(
                                           TPrivateKeySettings{
                                               .TvmHost = "",
                                               .TvmCacheDir = GetOutputPath() / "bad",
                                               .CacheTtl = TDuration::Seconds(0),
                                               .Env = NTvmAuth::EBlackboxEnv::Prod,
                                           },
                                           tvmClient),
                                       yexception, "Failed to read private keys from TVM API: ");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TPrivateKey(
                                           TPrivateKeySettings{
                                               .TvmHost = "",
                                               .EnableDiskCache = false,
                                               .Env = NTvmAuth::EBlackboxEnv::Prod,
                                           },
                                           tvmClient),
                                       yexception, "Failed to read private keys from TVM API: ");

        UNIT_ASSERT_NO_EXCEPTION(TPrivateKey(
            TPrivateKeySettings{
                .TvmHost = "",
                .TvmCacheDir = GetOutputPath() / "ok",
                .CacheTtl = TDuration::Days(5),
                .PreferredKeyIdx = 7,
                .Period = TDuration::Days(1),
                .RetryPeriod = TDuration::Hours(1),
                .Env = NTvmAuth::EBlackboxEnv::Test,
            },
            tvmClient));

        TPrivateKey pk(
            TPrivateKeySettings{
                .TvmHost = "",
                .TvmCacheDir = GetOutputPath() / "ok",
                .PreferredKeyIdx = 7,
                .Period = TDuration::Days(1),
                .RetryPeriod = TDuration::Hours(1),
                .Env = NTvmAuth::EBlackboxEnv::Prod,
            },
            tvmClient);

        TPrivateKey::TKey k = pk.GetKey();
        UNIT_ASSERT_EQUAL(11, k->GetId());
        UNIT_ASSERT_NO_EXCEPTION(pk.SelfCheck());
        UNIT_ASSERT_EXCEPTION_CONTAINS(pk.SelfCheck(10),
                                       yexception,
                                       "Basic check is failed: Expired ticket");
    }

    Y_UNIT_TEST(InitializationWithApi) {
        TPortManager pm;
        ui16 port = pm.GetPort(80);
        NMock::TMockServer server(port, []() { return new TGetPrivateKeys; });

        PrepareCache(TInstant::Now() - TDuration::Hours(2));
        UNIT_ASSERT_NO_EXCEPTION(
            TPrivateKey(
                TPrivateKeySettings{
                    .TvmHost = "localhost",
                    .TvmPort = port,
                    .TvmCacheDir = GetOutputPath() / "bad",
                    .EnableDiskCache = false,
                    .PreferredKeyIdx = 7,
                    .Period = TDuration::Days(1),
                    .RetryPeriod = TDuration::Hours(1),
                    .Env = NTvmAuth::EBlackboxEnv::Prod,
                },
                CreateMockedUpdater("ok.private")));

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TPrivateKey(
                TPrivateKeySettings{
                    .TvmHost = "localhost",
                    .TvmPort = port,
                    .TvmCacheDir = GetOutputPath() / "ok",
                    .EnableDiskCache = false,
                    .PreferredKeyIdx = 7,
                    .Period = TDuration::Days(1),
                    .RetryPeriod = TDuration::Hours(1),
                    .Env = NTvmAuth::EBlackboxEnv::Prod,
                },
                CreateMockedUpdater("bad.private")),
            yexception, "Failed to parse protobuf with private keys from TVM");

        TPrivateKey key(
            TPrivateKeySettings{
                .TvmHost = "localhost",
                .TvmPort = port,
                .TvmCacheDir = "/missing",
                .PreferredKeyIdx = 7,
                .Period = TDuration::Days(1),
                .RetryPeriod = TDuration::Hours(1),
                .Env = NTvmAuth::EBlackboxEnv::Prod,
            },
            CreateMockedUpdater("ok.private"));

        UNIT_ASSERT_VALUES_EQUAL(key.GetJugglerStatus().StatusCode(), NJuggler::ECode::Critical);
        UNIT_ASSERT_STRING_CONTAINS(key.GetJugglerStatus().Message(), "Private keys have not been updated from TVM API for");
    }

    Y_UNIT_TEST(JugglerStatus) {
        PrepareCache();
        std::shared_ptr<NTvmAuth::TTvmClient> tvmClient = CreateMockedUpdater("asdf");

        TPrivateKey ok(
            TPrivateKeySettings{
                .TvmHost = "",
                .TvmCacheDir = GetOutputPath() / "ok",
                .PreferredKeyIdx = 7,
                .Period = TDuration::Days(1),
                .RetryPeriod = TDuration::Hours(1),
                .Env = NTvmAuth::EBlackboxEnv::Prod,
            },
            tvmClient);

        UNIT_ASSERT_VALUES_EQUAL(ok.GetJugglerStatus(), NJuggler::TStatus(NJuggler::ECode::Ok, "OK"));

        TPrivateKey warning(
            TPrivateKeySettings{
                .TvmHost = "",
                .TvmCacheDir = GetOutputPath() / "ok",
                .JugglerWarningTimeout = TDuration::Seconds(0),
                .PreferredKeyIdx = 7,
                .Period = TDuration::Days(1),
                .RetryPeriod = TDuration::Hours(1),
                .Env = NTvmAuth::EBlackboxEnv::Prod,
            },
            tvmClient);

        UNIT_ASSERT_VALUES_EQUAL(warning.GetJugglerStatus().StatusCode(), NJuggler::ECode::Warning);
        UNIT_ASSERT_STRING_CONTAINS(warning.GetJugglerStatus().Message(), "Private keys have not been updated from TVM API for");

        TPrivateKey critical(
            TPrivateKeySettings{
                .TvmHost = "",
                .TvmCacheDir = GetOutputPath() / "ok",
                .JugglerWarningTimeout = TDuration::Seconds(0),
                .JugglerCriticalTimeout = TDuration::Seconds(0),
                .PreferredKeyIdx = 7,
                .Period = TDuration::Days(1),
                .RetryPeriod = TDuration::Hours(1),
                .Env = NTvmAuth::EBlackboxEnv::Prod,
            },
            tvmClient);

        UNIT_ASSERT_VALUES_EQUAL(critical.GetJugglerStatus().StatusCode(), NJuggler::ECode::Critical);
        UNIT_ASSERT_STRING_CONTAINS(critical.GetJugglerStatus().Message(), "Private keys have not been updated from TVM API for");
    }
}
