#include "signer.h"

#include <library/cpp/tvmauth/src/parser.h>
#include <library/cpp/tvmauth/src/utils.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

namespace NPassport::NTicketSigner {
    static const TString VERSION_ = "3";
    template <class Res, class Proto>
    Res SerializeStrV3(Proto& proto,
                       const NTvmAuth::NRw::TRwPrivateKey& key,
                       time_t expirationTime,
                       const TStringBuf serviceFlag) {
        proto.set_keyid(key.GetId());
        proto.set_expirationtime(expirationTime);
        TString protoStr = proto.SerializeAsString();

        Res res;
        const size_t SIGN_SIZE = 128;
        const size_t CONST_LITERALS = 8;
        res.reserve((protoStr.size() + SIGN_SIZE) * 4 / 3 + CONST_LITERALS);

        using namespace NTvmAuth;
        res.append(VERSION_).push_back(TParserTickets::DELIM);
        res.append(serviceFlag).push_back(TParserTickets::DELIM);
        res.append(NUtils::Bin2base64url(protoStr)).push_back(TParserTickets::DELIM);
        res.append(NUtils::Bin2base64url(key.SignTicket(res)));
        return res;
    }

    void TServiceSigner::SetSides(NTvmAuth::TTvmId src, NTvmAuth::TTvmId dst) {
        Proto_.mutable_service()->set_srcclientid(src);
        Proto_.mutable_service()->set_dstclientid(dst);
    }

    void TServiceSigner::AddScope(const TString& scope) {
        Proto_.mutable_service()->add_scopes(scope.c_str(), scope.size());
    }

    void TServiceSigner::SetIssuerUid(ui64 uid) {
        Proto_.mutable_service()->set_issueruid(uid);
    }

    TString TServiceSigner::SerializeV3(const NTvmAuth::NRw::TRwPrivateKey& key,
                                        time_t expireTime) {
        if (!Proto_.mutable_service()->has_dstclientid() || !Proto_.mutable_service()->has_srcclientid()) {
            throw yexception() << "Service ticket body is incomplete";
        }

        return SerializeStrV3<TString>(Proto_, key, expireTime, NTvmAuth::TParserTickets::ServiceFlag());
    }

    void TUserSigner::AddUid(ui64 uid) {
        if (!Enabled_) {
            return;
        }
        Uids_.insert(uid);
    }

    void TUserSigner::SetDefaultUid(ui64 uid) {
        if (!Enabled_) {
            return;
        }
        DefaultUid_ = uid;
    }

    void TUserSigner::AddScope(const TString& scope) {
        if (!Enabled_) {
            return;
        }
        Scopes_.insert(scope);
    }

    void TUserSigner::SetEntryPoint(NTvmAuth::TTvmId src) {
        if (!Enabled_) {
            return;
        }
        EntryPoint_ = src;
    }

    void TUserSigner::SetEnv(NTvmAuth::EBlackboxEnv env) {
        if (!Enabled_) {
            return;
        }

        using namespace NTvmAuth;
        switch (env) {
            case EBlackboxEnv::Prod:
                Env_ = tvm_keys::Prod;
                break;
            case EBlackboxEnv::Test:
                Env_ = tvm_keys::Test;
                break;
            case EBlackboxEnv::ProdYateam:
                Env_ = tvm_keys::ProdYateam;
                break;
            case EBlackboxEnv::TestYateam:
                Env_ = tvm_keys::TestYateam;
                break;
            case EBlackboxEnv::Stress:
                Env_ = tvm_keys::Stress;
                break;
        }
    }

    TString TUserSigner::SerializeV3(const NTvmAuth::NRw::TRwPrivateKey& key,
                                     time_t expireTime) const {
        if (!Enabled_) {
            throw yexception() << "UserSigner is not enabled";
        }

        CheckCompleteness();

        ticket2::Ticket proto;

        for (const ui64 u : Uids_) {
            proto.mutable_user()->add_users()->set_uid(u);
        }
        proto.mutable_user()->set_defaultuid(*DefaultUid_);

        for (const TString& scope : Scopes_) {
            proto.mutable_user()->add_scopes(scope.c_str(), scope.size());
        }

        proto.mutable_user()->set_entrypoint(*EntryPoint_);
        proto.mutable_user()->set_env(*Env_);

        return SerializeStrV3<TString>(proto, key, expireTime, NTvmAuth::TParserTickets::UserFlag());
    }

    size_t TUserSigner::size() const {
        size_t res = 8 * Uids_.size();

        for (const TString& s : Scopes_) {
            res += s.size();
        }

        res += sizeof(*this);

        return res;
    }

    size_t TUserSigner::Hash() const {
        size_t res = 0;

        for (ui64 u : Uids_) {
            res += std::hash<ui64>()(u);
        }
        if (DefaultUid_) {
            res += std::hash<ui64>()(*DefaultUid_);
        }

        for (const TString& s : Scopes_) {
            res += std::hash<TString>()(s);
        }

        if (EntryPoint_) {
            res += std::hash<ui32>()(*EntryPoint_);
        }

        return res;
    }

    void TUserSigner::CheckCompleteness() const {
        if (!DefaultUid_ || Uids_.empty() || !EntryPoint_ || !Env_) {
            throw yexception() << "User ticket body is incomplete";
        }

        if (*DefaultUid_ > 0 && Uids_.find(*DefaultUid_) == Uids_.end()) {
            throw yexception() << "User ticket has not default uid in missing uids";
        }
    }
}
