#pragma once

#include <library/cpp/tvmauth/checked_user_ticket.h>
#include <library/cpp/tvmauth/ticket_status.h>
#include <library/cpp/tvmauth/src/protos/ticket2.pb.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

#include <optional>
#include <set>
#include <unordered_map>

namespace NPassport::NTicketSigner {
    class TServiceSigner {
    public:
        void SetSides(NTvmAuth::TTvmId src, NTvmAuth::TTvmId dst);
        void AddScope(const TString& scope); // should be already sorted
        void SetIssuerUid(ui64 uid);

        TString SerializeV3(const NTvmAuth::NRw::TRwPrivateKey& key,
                            time_t expireTime);

    private:
        ticket2::Ticket Proto_;
    };

    class TUserSigner: TMoveOnly {
    public:
        TUserSigner(bool enabled = true)
            : Enabled_(enabled)
        {
        }

        TUserSigner(TUserSigner&&) = default;
        TUserSigner& operator=(TUserSigner&&) = default;

        void AddUid(ui64 uid);
        void SetDefaultUid(ui64 uid);
        void AddScope(const TString& scope);
        void SetEntryPoint(NTvmAuth::TTvmId src);
        void SetEnv(NTvmAuth::EBlackboxEnv env);

        TString SerializeV3(const NTvmAuth::NRw::TRwPrivateKey& key,
                            time_t expireTime) const;

        bool Enabled() const {
            return Enabled_;
        }

        size_t size() const;
        size_t Hash() const;

        bool operator==(const TUserSigner& o) const {
            return Uids_ == o.Uids_ &&
                   DefaultUid_ == o.DefaultUid_ &&
                   Scopes_ == o.Scopes_ &&
                   EntryPoint_ == o.EntryPoint_ &&
                   Env_ == o.Env_ &&
                   Enabled_ == o.Enabled_;
        }

        explicit operator bool() const {
            CheckCompleteness();
            return true;
        }

    private:
        void CheckCompleteness() const;

    private:
        std::set<ui64> Uids_;
        std::optional<ui64> DefaultUid_;
        std::set<TString> Scopes_; // sorting is required
        std::optional<NTvmAuth::TTvmId> EntryPoint_;
        std::optional<tvm_keys::BbEnvType> Env_;

        bool Enabled_;
    };
}

namespace std {
    template <>
    struct hash<NPassport::NTicketSigner::TUserSigner> {
        std::size_t operator()(const NPassport::NTicketSigner::TUserSigner& t) const {
            return t.Hash();
        }
    };
}
