#include <passport/infra/libs/cpp/tvm/signer/signer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/unittest.h>
#include <library/cpp/tvmauth/deprecated/service_context.h>
#include <library/cpp/tvmauth/deprecated/user_context.h>
#include <library/cpp/tvmauth/src/utils.h>

using namespace NPassport::NTicketSigner;
using namespace NTvmAuth;

Y_UNIT_TEST_SUITE(TvmSigner) {
    static const TString SECRET = "GRMJrKnj4fOVnvOqe-WyD1";

    Y_UNIT_TEST(ServiceOk) {
        const TTvmId src = 12;
        const TTvmId dst = 28;
        TServiceSigner b;
        b.SetSides(src, dst);
        b.AddScope("bb:sess");
        b.AddScope("bb:sess2");
        b.SetIssuerUid(123);

        NRw::TRwPrivateKey priv(NUtils::Base64url2bin("MIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYg"),
                                16);
        TString res = b.SerializeV3(priv, std::numeric_limits<time_t>::max());

        const TString CONST_OK = "3:serv:CBAQ__________9_IhkIDBAcGgdiYjpzZXNzGghiYjpzZXNzMiB7:";
        UNIT_ASSERT_VALUES_EQUAL(CONST_OK, res.substr(0, res.rfind(':') + 1));

        TServiceContext ctx(SECRET, dst, NUnittest::TVMKNIFE_PUBLIC_KEYS);
        TCheckedServiceTicket t = ctx.Check(res);
        UNIT_ASSERT(t);
        UNIT_ASSERT_VALUES_EQUAL((int)ETicketStatus::Ok, (int)t.GetStatus());
        UNIT_ASSERT_VALUES_EQUAL(src, t.GetSrc());

        ctx = TServiceContext(SECRET, dst + 1, NUnittest::TVMKNIFE_PUBLIC_KEYS);
        t = ctx.Check(res);
        UNIT_ASSERT(!t);
        UNIT_ASSERT_VALUES_EQUAL((int)ETicketStatus::InvalidDst, (int)t.GetStatus());
        UNIT_ASSERT_EXCEPTION_CONTAINS(t.GetSrc(), yexception, "Method cannot be used in non-valid ticket");
    }

    Y_UNIT_TEST(ServicePartiallyInited) {
        TServiceSigner b;
        b.AddScope("bb:sess");
        b.AddScope("bb:sess2");

        NRw::TRwPrivateKey priv(NUtils::Base64url2bin("MIIEmwKCAQBwsRd4frsVARIVSfj_vCdfvA3Q9SsGhSybdBDhbm8L6rPqxdoSNLCdNXzDWj7Ppf0o8uWHMxC-5Lfw0I18ri68nhm9-ndixcnbn6ti1uetgkc28eiEP6Q8ILD_JmkynbUl1aKDNAa5XsK2vFSEX402uydRomsTn46kRY23hfqcIi0ohh5VxIrpclRsRZus0JFu-RJzhqTbKYV4y4dglWPGHh5BuTv9k_Oh0_Ra8Xp5Rith5vjaKZUQ5Hyh9UtBYTkNWdvXP9OpmbiLVeRLuMzBm4HEFHDwMZ1h6LSVP-wB_spJPaMLTn3Q3JIHe-wGBYRWzU51RRYDqv4O_H12w5C1AoGBALAwCQ7fdAPG1lGclL7iWFjUofwPCFwPyDjicDT_MRRu6_Ta4GjqOGO9zuOp0o_ePgvR-7nA0fbaspM4LZNrPZwmoYBCJMtKXetg68ylu2DO-RRSN2SSh1AIZSA_8UTABk69bPzNL31j4PyZWxrgZ3zP9uZvzggveuKt5ZhCMoB7AoGBAKO9oC2AZjLdh2RaEFotTL_dY6lVcm38VA6PnigB8gB_TMuSrd4xtRw5BxvHpOCnBcUAJE0dN4_DDe5mrotKYMD2_3_lcq9PaLZadrPDCSDL89wtoVxNQNAJTqFjBFXYNu4Ze63lrsqg45TF5XmVRemyBHzXw3erd0pJaeoUDaSPAoGAJhGoHx_nVw8sDoLzeRkOJ1_6-uh_wVmVr6407_LPjrrySEq-GiYu43M3-QDp8J_J9e3S1Rpm4nQX2bEf5Gx9n4wKz7Hp0cwkOqBOWhvrAu6YLpv59wslEtkx0LYcJy6yQk5mpU8l29rPO7b50NyLnfnE2za-9DyK038FKlr5VgICgYAUd7QFsAzGW7Dsi0ILRamX-6x1Kq5Nv4qB0fPFAD5AD-mZclW7xjajhyDjePScFOC4oASJo6bx-GG9zNXRaUwYHt_v_K5V6e0Wy07WeGEkGX57hbQriagaASnULGCKuwbdwy91vLXZVBxymLyvMqi9NkCPmvhu9W7pSS09QoG0kgKBgBYGASHb7oB42sozkpfcSwsalD-B4QuB-QccTgaf5iKN3X6bXA0dRwx3udx1OlH7x8F6P3c4Gj7bVlJnBbJtZ7OE1DAIRJlpS71sHXmUt2wZ3yKKRuySUOoBDKQH_iiYAMnXrZ-Zpe-sfB-TK2NcDO-Z_tzN-cEF71xVvLMIRlAPAoGAdeikZPh1O57RxnVY72asiMRZheMBhK-9uSNPyYEZv3bUnIjg4XdMYStF2yTHNu014XvkDSQTe-drv2BDs9ExKplM4xFOtDtPQQ3mMB3GoK1qVhM_9n1QEElreurMicahkalnPo6tU4Z6PFL7PTpjRnCN67lJp0J0fxNDL13YSagCgYBA9VJrMtPjzcAx5ZCIYJjrYUPqEG_ttQN2RJIHN3MVpdpLAMIgX3tnlfyLwQFVKK45D1JgFa_1HHcxTWGtdIX4nsIjPWt-cWCCCkkw9rM5_Iqcb-YLSood6IP2OK0w0XLD1STnFRy_BRwdjPbGOYmp6YrJDZAlajDkFSdRvsz9Vg=="),
                                13);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "Service ticket body is incomplete");
    }

    Y_UNIT_TEST(UserOk) {
        TUserSigner b;
        b.AddUid(123);
        b.AddUid(456);
        b.SetDefaultUid(456);
        b.AddScope("bb:sess2");
        b.AddScope("bb:sess");
        b.SetEntryPoint(18);
        b.SetEnv(NTvmAuth::EBlackboxEnv::Test);
        UNIT_ASSERT(b.Enabled());

        NRw::TRwPrivateKey priv(NUtils::Base64url2bin("MIICVAKBgQCFM28TpJ13_6zbde_4q_CZVr67ZwlrEHdTAJ5eZoAK72rw7SngWB84ufwJOwx5S87gOmBPFte-RcSxSqTdoZM_pfjO5bVEnfDdQbyUYSuLeln31nwv9cD8qAG3kjdmG9Y8oLWSUPMBqFx3CUFF4jIrd3hTcspcLCRY9YMwEeCQnQJBAKV2lHRBYjFkrP0YKjh2ORfMWa_ZVsyK-9ZJzVf-SZdnmQNGpL0cKcf3lUOXeMxOTCC3HfwChyZU91Cv5GdVmwMCQQDOFa6pInktP9QbA1vnZgl-PLzAtJBrYm1JLxNEf2-yq1INmkXXjk7-Zy3mHBph3AXkU2J1fQt8YhMQ020hVIPfAkB4W6hR7PH0YVgJhE6Hnj3U_8bEmdvPD7URGtvdkqNM-cpR6WCHayt2WrRaCRslVT0344sTHa3kyyPyt2rFrCq8AkAZwrXVJE8lp_qDYGt87MEvx5eYFpINbE2pJeJoj-32VWpBs0i68cnfzOW8w4NMO4C8imxOr6FvjEJiGm2kKpB8AkAUrtKOiCxGLJWfowVHDsci-Ys1-yrZkV96yTmq_8ky7PMgaNSXo4U4_vKocu8ZicmEFuO_gFDkyp7qFfyM6rNgAkBOjawH30QJSX34KKqyNDcCia9GyNyai-67TIS7DVvEw_uKNPm6vNzlaIGI_kNo0sHNCy0QtQBKo2jzsC230BVdAkA8m8iZhL-98cA9VatBWTKV9VCMAjlnqtJuxDC9B-22ybhoza_rWnjWyvE0oJFzifFjeT1Gw6gGpN7_EDBRBHVE"),
                                13);
        TString str = b.SerializeV3(priv, std::numeric_limits<time_t>::max());

        const TString CONST_OK = "3:user:CA0Q__________9_GiMKAgh7CgMIyAMQyAMaB2JiOnNlc3MaCGJiOnNlc3MyIBIoAQ:";
        UNIT_ASSERT_VALUES_EQUAL(CONST_OK, str.substr(0, str.rfind(':') + 1));

        TUserContext ctx(EBlackboxEnv::Test, NUnittest::TVMKNIFE_PUBLIC_KEYS);
        TCheckedUserTicket res = ctx.Check(str);

        UNIT_ASSERT(res);
        UNIT_ASSERT_VALUES_EQUAL((int)ETicketStatus::Ok, (int)res.GetStatus());
        UNIT_ASSERT_EQUAL(TUids({123, 456}), res.GetUids());
        UNIT_ASSERT_VALUES_EQUAL(456, res.GetDefaultUid());
        UNIT_ASSERT_EQUAL(TScopes({"bb:sess", "bb:sess2"}), res.GetScopes());

        b.SetEnv(NTvmAuth::EBlackboxEnv::Prod);
        b.SetEnv(NTvmAuth::EBlackboxEnv::ProdYateam);
        b.SetEnv(NTvmAuth::EBlackboxEnv::TestYateam);
        b.SetEnv(NTvmAuth::EBlackboxEnv::Stress);
    }

    Y_UNIT_TEST(UserPartiallyInited) {
        TUserSigner b;
        b.SetDefaultUid(123);

        NRw::TRwPrivateKey priv(NUtils::Base64url2bin("MIIEmwKCAQBwsRd4frsVARIVSfj_vCdfvA3Q9SsGhSybdBDhbm8L6rPqxdoSNLCdNXzDWj7Ppf0o8uWHMxC-5Lfw0I18ri68nhm9-ndixcnbn6ti1uetgkc28eiEP6Q8ILD_JmkynbUl1aKDNAa5XsK2vFSEX402uydRomsTn46kRY23hfqcIi0ohh5VxIrpclRsRZus0JFu-RJzhqTbKYV4y4dglWPGHh5BuTv9k_Oh0_Ra8Xp5Rith5vjaKZUQ5Hyh9UtBYTkNWdvXP9OpmbiLVeRLuMzBm4HEFHDwMZ1h6LSVP-wB_spJPaMLTn3Q3JIHe-wGBYRWzU51RRYDqv4O_H12w5C1AoGBALAwCQ7fdAPG1lGclL7iWFjUofwPCFwPyDjicDT_MRRu6_Ta4GjqOGO9zuOp0o_ePgvR-7nA0fbaspM4LZNrPZwmoYBCJMtKXetg68ylu2DO-RRSN2SSh1AIZSA_8UTABk69bPzNL31j4PyZWxrgZ3zP9uZvzggveuKt5ZhCMoB7AoGBAKO9oC2AZjLdh2RaEFotTL_dY6lVcm38VA6PnigB8gB_TMuSrd4xtRw5BxvHpOCnBcUAJE0dN4_DDe5mrotKYMD2_3_lcq9PaLZadrPDCSDL89wtoVxNQNAJTqFjBFXYNu4Ze63lrsqg45TF5XmVRemyBHzXw3erd0pJaeoUDaSPAoGAJhGoHx_nVw8sDoLzeRkOJ1_6-uh_wVmVr6407_LPjrrySEq-GiYu43M3-QDp8J_J9e3S1Rpm4nQX2bEf5Gx9n4wKz7Hp0cwkOqBOWhvrAu6YLpv59wslEtkx0LYcJy6yQk5mpU8l29rPO7b50NyLnfnE2za-9DyK038FKlr5VgICgYAUd7QFsAzGW7Dsi0ILRamX-6x1Kq5Nv4qB0fPFAD5AD-mZclW7xjajhyDjePScFOC4oASJo6bx-GG9zNXRaUwYHt_v_K5V6e0Wy07WeGEkGX57hbQriagaASnULGCKuwbdwy91vLXZVBxymLyvMqi9NkCPmvhu9W7pSS09QoG0kgKBgBYGASHb7oB42sozkpfcSwsalD-B4QuB-QccTgaf5iKN3X6bXA0dRwx3udx1OlH7x8F6P3c4Gj7bVlJnBbJtZ7OE1DAIRJlpS71sHXmUt2wZ3yKKRuySUOoBDKQH_iiYAMnXrZ-Zpe-sfB-TK2NcDO-Z_tzN-cEF71xVvLMIRlAPAoGAdeikZPh1O57RxnVY72asiMRZheMBhK-9uSNPyYEZv3bUnIjg4XdMYStF2yTHNu014XvkDSQTe-drv2BDs9ExKplM4xFOtDtPQQ3mMB3GoK1qVhM_9n1QEElreurMicahkalnPo6tU4Z6PFL7PTpjRnCN67lJp0J0fxNDL13YSagCgYBA9VJrMtPjzcAx5ZCIYJjrYUPqEG_ttQN2RJIHN3MVpdpLAMIgX3tnlfyLwQFVKK45D1JgFa_1HHcxTWGtdIX4nsIjPWt-cWCCCkkw9rM5_Iqcb-YLSood6IP2OK0w0XLD1STnFRy_BRwdjPbGOYmp6YrJDZAlajDkFSdRvsz9Vg=="),
                                13);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "User ticket body is incomplete");

        b.AddScope("bb:sess");
        b.AddScope("bb:sess2");

        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "User ticket body is incomplete");

        b.AddUid(123);

        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "User ticket body is incomplete");

        b.SetEntryPoint(123);

        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "User ticket body is incomplete");

        b.AddUid(456);
        b.SetDefaultUid(789);
        b.SetEntryPoint(18);
        b.SetEnv(NTvmAuth::EBlackboxEnv::Test);

        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "User ticket has not default uid in missing uids");
    }

    Y_UNIT_TEST(UserDisabled) {
        NRw::TRwPrivateKey priv(NUtils::Base64url2bin("MIIEmwKCAQBwsRd4frsVARIVSfj_vCdfvA3Q9SsGhSybdBDhbm8L6rPqxdoSNLCdNXzDWj7Ppf0o8uWHMxC-5Lfw0I18ri68nhm9-ndixcnbn6ti1uetgkc28eiEP6Q8ILD_JmkynbUl1aKDNAa5XsK2vFSEX402uydRomsTn46kRY23hfqcIi0ohh5VxIrpclRsRZus0JFu-RJzhqTbKYV4y4dglWPGHh5BuTv9k_Oh0_Ra8Xp5Rith5vjaKZUQ5Hyh9UtBYTkNWdvXP9OpmbiLVeRLuMzBm4HEFHDwMZ1h6LSVP-wB_spJPaMLTn3Q3JIHe-wGBYRWzU51RRYDqv4O_H12w5C1AoGBALAwCQ7fdAPG1lGclL7iWFjUofwPCFwPyDjicDT_MRRu6_Ta4GjqOGO9zuOp0o_ePgvR-7nA0fbaspM4LZNrPZwmoYBCJMtKXetg68ylu2DO-RRSN2SSh1AIZSA_8UTABk69bPzNL31j4PyZWxrgZ3zP9uZvzggveuKt5ZhCMoB7AoGBAKO9oC2AZjLdh2RaEFotTL_dY6lVcm38VA6PnigB8gB_TMuSrd4xtRw5BxvHpOCnBcUAJE0dN4_DDe5mrotKYMD2_3_lcq9PaLZadrPDCSDL89wtoVxNQNAJTqFjBFXYNu4Ze63lrsqg45TF5XmVRemyBHzXw3erd0pJaeoUDaSPAoGAJhGoHx_nVw8sDoLzeRkOJ1_6-uh_wVmVr6407_LPjrrySEq-GiYu43M3-QDp8J_J9e3S1Rpm4nQX2bEf5Gx9n4wKz7Hp0cwkOqBOWhvrAu6YLpv59wslEtkx0LYcJy6yQk5mpU8l29rPO7b50NyLnfnE2za-9DyK038FKlr5VgICgYAUd7QFsAzGW7Dsi0ILRamX-6x1Kq5Nv4qB0fPFAD5AD-mZclW7xjajhyDjePScFOC4oASJo6bx-GG9zNXRaUwYHt_v_K5V6e0Wy07WeGEkGX57hbQriagaASnULGCKuwbdwy91vLXZVBxymLyvMqi9NkCPmvhu9W7pSS09QoG0kgKBgBYGASHb7oB42sozkpfcSwsalD-B4QuB-QccTgaf5iKN3X6bXA0dRwx3udx1OlH7x8F6P3c4Gj7bVlJnBbJtZ7OE1DAIRJlpS71sHXmUt2wZ3yKKRuySUOoBDKQH_iiYAMnXrZ-Zpe-sfB-TK2NcDO-Z_tzN-cEF71xVvLMIRlAPAoGAdeikZPh1O57RxnVY72asiMRZheMBhK-9uSNPyYEZv3bUnIjg4XdMYStF2yTHNu014XvkDSQTe-drv2BDs9ExKplM4xFOtDtPQQ3mMB3GoK1qVhM_9n1QEElreurMicahkalnPo6tU4Z6PFL7PTpjRnCN67lJp0J0fxNDL13YSagCgYBA9VJrMtPjzcAx5ZCIYJjrYUPqEG_ttQN2RJIHN3MVpdpLAMIgX3tnlfyLwQFVKK45D1JgFa_1HHcxTWGtdIX4nsIjPWt-cWCCCkkw9rM5_Iqcb-YLSood6IP2OK0w0XLD1STnFRy_BRwdjPbGOYmp6YrJDZAlajDkFSdRvsz9Vg=="),
                                13);

        TUserSigner b(false);
        UNIT_ASSERT(!b.Enabled());
        b.AddUid(123);
        b.SetDefaultUid(789);
        b.AddScope("bb:sess");
        b.SetEntryPoint(18);
        b.SetEnv(NTvmAuth::EBlackboxEnv::Test);
        UNIT_ASSERT_EXCEPTION_CONTAINS(b.SerializeV3(priv, 100500), yexception, "UserSigner is not enabled");
    }

    Y_UNIT_TEST(UserEq) {
        TUserSigner b(true);
        b.AddUid(123);
        b.SetDefaultUid(789);
        b.AddScope("bb:sess");
        b.SetEntryPoint(18);
        b.SetEnv(NTvmAuth::EBlackboxEnv::Test);

        TUserSigner b2;
        b2.AddUid(123);
        b2.SetDefaultUid(789);
        b2.AddScope("bb:sess");
        b2.SetEntryPoint(18);
        b2.SetEnv(NTvmAuth::EBlackboxEnv::Test);

        UNIT_ASSERT_EQUAL(b2, b);
    }
}
