#pragma once

#include "diff.h"

#include <memory>
#include <vector>

namespace NPassport::NUnistat {
    template <typename T = NBase::TSignalDiff<ui64>>
    class TSignalHistogram {
        static_assert(T::TIsDiffSignalTrait::value, "histogram can work only with diff signals");
        static_assert(std::is_same_v<ui64, typename T::TCommonType>, "histogram can work only with ui64 counter");

    public:
        using TValue = ui64;
        using TBounds = std::vector<TValue>;

        struct TPair {
            TValue LeftBound;
            std::unique_ptr<T> Value;
        };
        using TValues = std::vector<TPair>;

        TSignalHistogram(const TString& name, TBounds&& bounds)
            : Name_(name + "_dhhh")
        {
            bounds.push_back(0);
            std::sort(bounds.begin(), bounds.end());
            bounds.erase(std::unique(bounds.begin(), bounds.end()), bounds.end());

            for (const unsigned ms : bounds) {
                Values_.push_back(TPair{ms, std::make_unique<T>()});
            }
        }

        void AddValue(TValue value) {
            auto pos = std::lower_bound(
                Values_.rbegin(),
                Values_.rend(),
                TPair{value, nullptr},
                [](const auto& p1, const auto& p2) { return p1.LeftBound > p2.LeftBound; });
            if (pos == Values_.rend()) {
                pos = Values_.rbegin();
            }
            ++(*pos->Value);
        }

        void Flush() {
            if constexpr (std::is_same_v<T, NBase::TSignalBufferedDiff<ui64>>) {
                for (const auto& pair : Values_) {
                    pair.Value->Flush();
                }
            }
        }

        const TString& GetName() const {
            return Name_;
        }

        const TValues& GetValues() const {
            return Values_;
        }

    private:
        const TString Name_;
        TValues Values_;
    };

    using TSignalBufferedHistogram = TSignalHistogram<NBase::TSignalBufferedDiff<ui64>>;
}
