#include "exponential_backoff.h"

#include <util/random/normal.h>

namespace NPassport::NUtils {
    TExponentialBackoff::TExponentialBackoff(TDuration min, TDuration max, double factor, double jitter)
        : Min_(min)
        , Max_(max)
        , Factor_(factor)
        , Jitter_(jitter)
        , CurrentValue_(Min_)
    {
        Y_ENSURE(Factor_ > 1, "factor=" << Factor_ << ". Should be > 1");
        Y_ENSURE(Jitter_ >= 0 && Jitter_ < 1, "jitter should be in range [0, 1)");
    }

    TDuration TExponentialBackoff::Increase() {
        CurrentValue_ = std::min(CurrentValue_ * Factor_, Max_);

        double rnd = StdNormalRandom<double>();
        const bool isNegative = rnd < 0;
        rnd = std::abs(rnd);

        const TDuration diff = rnd * Jitter_ * CurrentValue_;
        if (isNegative) {
            CurrentValue_ -= diff;
        } else {
            CurrentValue_ += diff;
        }

        return CurrentValue_;
    }

    TDuration TExponentialBackoff::Decrease() {
        CurrentValue_ = std::max(CurrentValue_ / Factor_, Min_);
        return CurrentValue_;
    }

    void TExponentialBackoff::Sleep() {
        Ev_.WaitT(CurrentValue_);
    }

    void TExponentialBackoff::Interrupt() {
        Ev_.Signal();
    }
}
